package ru.org.amip.ambisync;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.philips.lighting.hue.sdk.PHHueSDK;
import com.philips.lighting.model.PHBridge;
import com.philips.lighting.model.PHBridgeResourcesCache;
import com.philips.lighting.model.PHLight;
import com.philips.lighting.model.PHLightState;
import com.tpvision.ambilightplushue.helpers.Algorithm;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.quartz.*;
import org.quartz.impl.StdSchedulerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.org.amip.ambisync.config.Lamp;
import ru.org.amip.ambisync.config.LampConfig;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;

import static ru.org.amip.ambisync.Main.conf;

/**
 * Date: 28.02.14 Time: 5:18
 *
 * @author serge
 */
public class AmbiSync {
  private static final Logger logger = LoggerFactory.getLogger(AmbiSync.class);

  private static AmbiSync instance = new AmbiSync();

  public static final int MAX_ERRORS = conf.getInt("tv.max.errors", 10);
  public static final int TV_TIMEOUT = conf.getInt("tv.timeout", 5000);

  private String              tvUrl;
  private HttpGet             httpget;
  private CloseableHttpClient client;

  private ResponseHandler<JsonObject> rh;

  private HashMap<String, LampState>    previousState = new HashMap<>();
  private HashMap<String, PHLightState> savedState    = new HashMap<>();
  private HashMap<String, String>       lampModel     = new HashMap<>();

  private PHHueSDK  phHueSDK;
  private Scheduler scheduler;

  private volatile int     syncErrors = 0;
  private volatile boolean running    = false;
  private volatile Timer             retryTimer;
  private volatile Map<String, Lamp> lamps;

  private long syncStopped;

  private boolean detectStandby;

  public synchronized long getSyncStopped() {
    return syncStopped;
  }

  public synchronized void setSyncStopped(long syncStopped) {
    this.syncStopped = syncStopped;
  }

  public boolean isRunning() {
    return running;
  }

  public static AmbiSync getInstance() {
    return instance;
  }

  public Scheduler getScheduler() {
    return scheduler;
  }

  public void setLampsConfig(final String config) {
    logger.info("Changing lamps config to '{}'", config);

    new Timer().schedule(new TimerTask() {
      @Override
      public void run() {
        boolean running = isRunning();
        if (running) {
          stopSync(false);
          Controller.delay(2000);
        }

        conf.setProperty("sync.preset", config);

        Main.saveConfig();

        lamps = new LampConfig().load(conf.getString("sync.preset"));

        if (running) {
          startSync(false);
        }
      }
    }, 0);
  }

  private AmbiSync() {
    phHueSDK = PHHueSDK.getInstance();
    lamps = new LampConfig().load(conf.getString("sync.preset"));
    detectStandby = conf.getBoolean("sync.detect.standby", true);

    final String tvIP = conf.getString("tv.ip");

    if (tvIP == null || tvIP.isEmpty()) {
      logger.error("Please edit 'conf/application.conf' and set 'tv.ip' property");
      new Thread(new Runnable() {
        @Override
        public void run() {
          Controller.delay(3000);
          System.exit(1);
        }
      }).start();
      return;
    }

    tvUrl = "http://" + tvIP + ":" + conf.getString("tv.port") + conf.getString("tv.ambi");
    logger.info("Ambilight TV URL: {}", tvUrl);

    RequestConfig.Builder requestBuilder = RequestConfig.custom();
    requestBuilder = requestBuilder.setConnectTimeout(TV_TIMEOUT);
    requestBuilder = requestBuilder.setConnectionRequestTimeout(TV_TIMEOUT);
    requestBuilder = requestBuilder.setSocketTimeout(TV_TIMEOUT);

    client = HttpClients.custom()
                        .setDefaultRequestConfig(requestBuilder.build())
                        .build();

    httpget = new HttpGet(tvUrl);

    rh = new ResponseHandler<JsonObject>() {
      @Override
      public JsonObject handleResponse(
        final HttpResponse response) throws IOException {
        StatusLine statusLine = response.getStatusLine();
        HttpEntity entity = response.getEntity();
        if (statusLine.getStatusCode() >= 300) {
          throw new HttpResponseException(
            statusLine.getStatusCode(),
            statusLine.getReasonPhrase());
        }
        if (entity == null) {
          throw new ClientProtocolException("Response contains no content");
        }

        JsonParser parser = new JsonParser();

        ContentType contentType = ContentType.getOrDefault(entity);
        Charset charset = contentType.getCharset();
        Reader reader = new InputStreamReader(entity.getContent(), charset);

        return (JsonObject) parser.parse(reader);
      }
    };

    prepareScheduler();
  }

  public TriggerKey schedulePeriodicJob(String name, Class<? extends Job> jobClass, int interval) throws SchedulerException {
    final JobDetail job = JobBuilder.newJob(jobClass).withIdentity(name).build();
    final SimpleTrigger trigger = TriggerBuilder
      .newTrigger()
      .withIdentity(name + "-trigger")
      .startNow()
      .withSchedule(SimpleScheduleBuilder.simpleSchedule().withIntervalInMilliseconds(interval).repeatForever()).build();

    scheduler.scheduleJob(job, trigger);
    return trigger.getKey();
  }

  private void prepareScheduler() {
    try {
      scheduler = StdSchedulerFactory.getDefaultScheduler();
      schedulePeriodicJob("syncAmbilight", AmbilightSyncJob.class, conf.getInt("sync.interval"));
      schedulePeriodicJob("forceSyncAmbilight", AmbilightForcedSyncJob.class, conf.getInt("forced.sync.interval"));
    } catch (SchedulerException e) {
      logger.error("Scheduler error", e);
    }
  }

  public void saveLampsState() {
    logger.info("Saving current lamps state...");

    final long updated = Controller.getInstance().getCacheUpdated();
    final long stopped = getSyncStopped();

    logger.debug("Cache updated: {}, sync stopped: {}", new Date(updated), new Date(stopped));

    boolean save = true;
    if (updated - 20000 < stopped) {
      logger.info("Save aborted, lamp state cache can contain ambilight values");
      save = false;
    }

    PHBridge bridge = phHueSDK.getSelectedBridge();
    PHBridgeResourcesCache cache = bridge.getResourceCache();

    List<PHLight> allLights = cache.getAllLights();

    if (save) {
      savedState.clear();
      lampModel.clear();
    }

    for (PHLight light : allLights) {
      final PHLightState state = new PHLightState(light.getLastKnownLightState());
      if (save) savedState.put(light.getIdentifier(), state);
      if (lamps.containsKey(light.getIdentifier())) {
        lampModel.put(light.getIdentifier(), light.getModelNumber());
      } else if (conf.getBoolean("sync.turn.off.other", false)) {
        PHLightState lightState = new PHLightState();
        lightState.setOn(false);
        bridge.updateLightState(light, lightState);
        Controller.delay();
      }
    }
  }

  public void restoreLampsState() {
    logger.info("Restoring last lamps state...");
    for (int i = 0; i < 2; i++) { // try twice, sometimes it doesn't work
      for (Map.Entry<String, PHLightState> entry : savedState.entrySet()) {
        phHueSDK.getSelectedBridge().updateLightState(entry.getKey(), entry.getValue(), null);
        Controller.delay();
      }
      Controller.delay(1000);
    }
  }

  public void setLampsState(boolean state) {
    logger.info("Turning {} lamps configured for synchronization...", state ? "on" : "off");

    PHBridge bridge = phHueSDK.getSelectedBridge();
    PHBridgeResourcesCache cache = bridge.getResourceCache();

    List<PHLight> allLights = cache.getAllLights();

    for (int i = 0; i < 2; i++) { // try twice, sometimes it doesn't work
      for (PHLight light : allLights) {
        logger.trace("{}", light);
        if (lamps.containsKey(light.getIdentifier())) {
          PHLightState lightState = new PHLightState();
          lightState.setOn(state);
          bridge.updateLightState(light, lightState);
          Controller.delay();
        }
      }
      Controller.delay(50);
    }
  }

  private class LampState {
    float x, y;
    int brightness;
    RGB color;

    private LampState(float x, float y, int brightness, RGB color) {
      this.x = x;
      this.y = y;
      this.brightness = brightness;
      this.color = color;
    }

    public float getX() {
      return x;
    }

    public float getY() {
      return y;
    }

    public int getBrightness() {
      return brightness;
    }

    public RGB getColor() {
      return color;
    }
  }

  public synchronized void update(boolean force) {
    try {
      final JsonObject ambilightData = client.execute(httpget, rh);
      syncErrors = 0;
      int standbyModeColor = 0;

      for (Map.Entry<String, Lamp> entry : lamps.entrySet()) {
        final RGB color = AmbilightUtil.getAverageColor(entry.getValue(), ambilightData);
        standbyModeColor += color.r + color.g + color.b;

        String lid = entry.getKey();
        LampState ps = previousState.get(lid);

        if (!force && ps != null && ps.getColor().equalsWithOffset(color, conf.getInt("rgb.offset"))) continue;

        AtomicInteger outBrightness = new AtomicInteger();
        final Lamp lamp = lamps.get(lid);

        String model = lampModel.get(lid);
        // bridge doesn't report the model, try to get it from the config
        if (model == null) {
          model = lamp.getModel();
        }
        if (model == null) {
          logger.warn("Lamp {} model is not detected, will default to LCT001", lid);
          model = "LCT001";
        }

        final float[] xy =
          Algorithm.convertRGBtoXY_final(
            color.r / 255F,
            color.g / 255F, color.b / 255F, 0, 0, 0,
            outBrightness,
            lamp.getImmersion(), lamp.getBrightness(), model);

        if (force || ps == null || ps.getX() != xy[0] || ps.getY() != xy[1] ||
            ps.getBrightness() != outBrightness.get()) {
          logger.trace("@{}@> {} => {},{},{}|{}", lid, color, xy[0], xy[1], outBrightness.get(), lamp.getTransition());

          PHLightState lightState = new PHLightState();
          lightState.setX(xy[0]);
          lightState.setY(xy[1]);
          lightState.setBrightness(outBrightness.get());
          lightState.setTransitionTime(lamp.getTransition());

          if (!running) return;
          phHueSDK.getSelectedBridge().updateLightState(lid, lightState, null);
          Controller.delay();
        }

        previousState.put(lid, new LampState(xy[0], xy[1], outBrightness.get(), color));
      }

      // try to detect standby mode and disable sync
      if (detectStandby && force && lamps.size() > 0 && standbyModeColor == 0) {
        throw new StandbyException("standby mode enabled");
      }
    } catch (Exception e) {
      logger.error("Can't get ambilight from TV: {}", e.getMessage());
      syncErrors++;
      if (syncErrors > MAX_ERRORS || e instanceof StandbyException) {
        if (!(e instanceof StandbyException))
          logger.error("Maximum errors count ({}) reached, stopping sync", MAX_ERRORS);

        stopSync(true);
        if (conf.getBoolean("sync.resume", true)) startSync(true);
      }
    }
  }

  public void startSync(final boolean check) {
    if (running) return;
    new Timer().schedule(new TimerTask() {
      @Override
      public void run() {
        try {
          if (check) {
            logger.info("Checking for ambilight data at {}", tvUrl);
            final JsonObject ambilightData = client.execute(httpget, rh);
            int standbyModeColor = 0;
            for (Map.Entry<String, Lamp> entry : lamps.entrySet()) {
              final RGB color = AmbilightUtil.getAverageColor(entry.getValue(), ambilightData);
              standbyModeColor += color.r + color.g + color.b;
            }
            if (standbyModeColor == 0) {
              throw new StandbyException();
            }
            logger.info("TV is available, starting sync!");
          }
          _startSync();
        } catch (Exception e) {
          final int retry = conf.getInt("tv.retry.interval", 60);

          if (e instanceof IOException) {
            logger.info("TV is not available, will try to ping it again in {} seconds", retry);
          } else if (e instanceof StandbyException) {
            logger.info("Ambilight data is all zeroes, standby mode is active");
          }

          retryTimer = new Timer();
          retryTimer.schedule(new TimerTask() {
            @Override
            public void run() {
              startSync(true);
            }
          }, retry * 1000);
        }
      }
    }, 0);
  }

  private void _startSync() {
    if (running) return;
    logger.info("Sync started");

    running = true;
    syncErrors = 0;

    if (conf.getBoolean("sync.restore")) saveLampsState();
    setLampsState(true);

    try {
      scheduler.start();
    } catch (SchedulerException e) {
      logger.error("Scheduler error", e);
    }
  }

  public void stopSync(final boolean abnormal) {
    if (retryTimer != null) retryTimer.cancel();

    if (!running) return;

    try {
      if (scheduler != null) scheduler.standby();
    } catch (SchedulerException e) {
      e.printStackTrace();
    }

    running = false;

    new Timer().schedule(new TimerTask() {
      @Override
      public void run() {
        logger.info("Sync stopped");
        setSyncStopped(System.currentTimeMillis());
        if (conf.getBoolean("sync.restore")) restoreLampsState();
        Controller.delay(1000);
        if (conf.getBoolean("sync.lamps.off.on.stop", false)) setLampsState(false);
        if (abnormal && conf.getBoolean("sync.lamps.off.on.error", true)) {
          setLampsState(false);
        }
      }
    }, 1000);
  }
}
