package ru.org.amip.ambisync.web;

import com.philips.lighting.hue.sdk.PHAccessPoint;
import com.philips.lighting.hue.sdk.PHHueSDK;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.org.amip.ambisync.AmbiSync;
import ru.org.amip.ambisync.Controller;
import ru.org.amip.ambisync.Version;
import ru.org.amip.ambisync.config.Command;
import ru.org.amip.ambisync.config.CommandPreset;
import ru.org.amip.ambisync.config.Preset;
import spark.Filter;
import spark.Request;
import spark.Response;
import spark.Route;
import spark.servlet.SparkApplication;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static ru.org.amip.ambisync.Main.conf;
import static spark.Spark.*;

/**
 * Date: 14.03.2014 Time: 14:34
 *
 * @author serge
 */
public class AmbiWeb implements SparkApplication {
  private static final Logger logger = LoggerFactory.getLogger(AmbiWeb.class);

  private AmbiSync   sync       = AmbiSync.getInstance();
  private Controller controller = Controller.getInstance();

  private String getStatus() {return sync.isRunning() ? "1" : "0";}

  @Override
  public void init() {
    logger.trace("Spark init");

    setPort(conf.getInt("web.port"));
    setIpAddress(conf.getString("web.ip"));

    enableCORS("*", "GET", "*");

    after(new Filter() {
      @Override
      public void handle(Request request, Response response) {
        response.header("Content-type", "text/plain");
      }
    });

    get(new Route("/") {
      @Override
      public Object handle(Request request, Response response) {
        return String.format("AmbiSync v%s (built: %s)\nStatus: %s", Version.getVersion(), Version.getTimeStamp(), getStatus());
      }
    });

    get(new Route("/status") {
      @Override
      public Object handle(Request request, Response response) {
        return sync.isRunning() ? "1" : "0";
      }
    });

    get(new Route("/stop") {
      @Override
      public Object handle(Request request, Response response) {
        stopSync();
        return getStatus();
      }
    });

    get(new Route("/start") {
      @Override
      public Object handle(Request request, Response response) {
        startSync();
        return getStatus();
      }
    });

    get(new Route("/toggle") {
      @Override
      public Object handle(Request request, Response response) {
        if (sync.isRunning()) {
          stopSync();
        } else {
          startSync();
        }
        return getStatus();
      }
    });

    get(new Route("/sync-preset/load/:config") {
      @Override
      public Object handle(Request request, Response response) {
        final String config = request.params(":config");
        sync.setLampsConfig(config);
        return config;
      }
    });

    get(new Route("/on") {
      @Override
      public Object handle(Request request, Response response) {
        sync.setLampsState(true);
        sync.restoreLampsState();
        return "on";
      }
    });

    get(new Route("/off") {
      @Override
      public Object handle(Request request, Response response) {
        sync.saveLampsState();
        sync.setLampsState(false);
        return "off";
      }
    });

    get(new Route("/state/save") {
      @Override
      public Object handle(Request request, Response response) {
        sync.saveLampsState();
        return "save";
      }
    });

    get(new Route("/state/load") {
      @Override
      public Object handle(Request request, Response response) {
        sync.restoreLampsState();
        return "load";
      }
    });

    get(new Route("/preset") {
      @Override
      public Object handle(Request request, Response response) {
        final String preset = Preset.getInstance().getLastPreset();
        return preset == null ? "null" : preset;
      }
    });

    get(new Route("/preset/list") {
      @Override
      public Object handle(Request request, Response response) {
        StringBuilder sb = new StringBuilder();
        final ArrayList<String> all = Preset.getInstance().getAll();
        for (String s : all) {
          sb.append(s);
          sb.append("\n");
        }
        return sb.toString().trim();
      }
    });

    get(new Route("/preset/load/:name") {
      @Override
      public Object handle(Request request, Response response) {
        final String name = request.params(":name");
        if (name != null && !name.isEmpty()) {
          if (Preset.getInstance().load(name)) {
            return name;
          }
        }
        return "error";
      }
    });

    get(new Route("/preset/load/:name/:delay") {
      @Override
      public Object handle(Request request, Response response) {
        final String name = request.params(":name");
        final long delay = Long.parseLong(request.params(":delay"));
        if (name != null && !name.isEmpty()) {
          Preset.getInstance().load(name, delay);
          return name + " in " + delay;
        }
        return "error";
      }
    });

    get(new Route("/preset/save/:name") {
      @Override
      public Object handle(Request request, Response response) {
        final String name = request.params(":name");
        if (name != null && !name.isEmpty()) {
          if (Preset.getInstance().save(name)) {
            return String.format("Saved preset: '%s'", name);
          }
        }
        return String.format("Error saving preset: %s", name);
      }
    });

    get(new Route("/bridge") {
      @Override
      public Object handle(Request request, Response response) {
        final List<PHAccessPoint> bridges = controller.getBridges();
        if (bridges.size() == 0) {
          controller.findBridges();
          return "No bridges found, refresh this page to search again";
        } else {
          StringBuilder sb = new StringBuilder();
          for (PHAccessPoint bridge : bridges) {
            sb.append(bridge.getIpAddress());
            sb.append(" [");
            sb.append(bridge.getMacAddress());
            sb.append("]\n");
          }
          return sb.toString();
        }
      }
    });

    get(new Route("/bridge/connect/:ip") {
      @Override
      public Object handle(Request request, Response response) {
        final String ip = request.params(":ip");
        return connectToBridge(ip);
      }
    });

    get(new Route("/bridge/connect") {
      @Override
      public Object handle(Request request, Response response) {
        return connectToBridge(null);
      }
    });

    get(new Route("/cc/:trigger") {
      @Override
      public Object handle(Request request, Response response) {
        final String trigger = request.params(":trigger");
        if (trigger != null && !trigger.isEmpty()) {
          if (Command.getInstance().handleCommand(trigger, getQueryParamsMap(request))) {
            return trigger;
          }
        }
        return "error";
      }
    });

    get(new Route("/cc-ctrl/reload") {
      @Override
      public Object handle(Request request, Response response) {
        return Command.getInstance().loadPresets();
      }
    });

    get(new Route("/cc-ctrl/list") {
      @Override
      public Object handle(Request request, Response response) {
        StringBuilder sb = new StringBuilder();
        final HashMap<String, CommandPreset> presets = Command.getInstance().getPresets();
        for (Map.Entry<String, CommandPreset> entry : presets.entrySet()) {
          sb.append(entry.getKey());
          sb.append(": ");
          sb.append(entry.getValue().description);
          sb.append("\n");
        }
        return sb.toString();
      }
    });
  }

  private static void enableCORS(final String origin, final String methods, final String headers) {
    options(new Route("/*") {
      @Override
      public Object handle(Request request, Response response) {
        String accessControlRequestHeaders = request.headers("Access-Control-Request-Headers");
        if (accessControlRequestHeaders != null) {
          response.header("Access-Control-Allow-Headers", accessControlRequestHeaders);
        }

        String accessControlRequestMethod = request.headers("Access-Control-Request-Method");
        if (accessControlRequestMethod != null) {
          response.header("Access-Control-Allow-Methods", accessControlRequestMethod);
        }

        return "OK";
      }
    });

    before(new Filter() {
      @Override
      public void handle(Request request, Response response) {
        response.header("Access-Control-Allow-Origin", origin);
        response.header("Access-Control-Request-Method", methods);
        response.header("Access-Control-Allow-Headers", headers);
        // Note: this may or may not be necessary in your particular application
        // response.type("application/json");
      }
    });
  }

  private Map<String, String> getQueryParamsMap(Request request) {
    final HashMap<String, String> map = new HashMap<>();
    for (String param : request.queryParams()) {
      map.put(param, request.queryParams(param));
    }
    return map;
  }

  private void startSync() {
    controller.setSyncEnabled(true);
    sync.startSync(false);
  }

  private void stopSync() {
    controller.setSyncEnabled(false);
    sync.stopSync(false);
  }

  private Object connectToBridge(String ip) {
    final List<PHAccessPoint> bridges = controller.getBridges();
    if (bridges.size() == 0) {
      controller.findBridges();
      return "No bridges found, refresh this page to search again";
    } else {
      for (PHAccessPoint bridge : bridges) {
        if (ip == null || ip.equalsIgnoreCase(bridge.getIpAddress())) {
          bridge.setUsername(conf.getString("bridge.username"));
          PHHueSDK.getInstance().connect(bridge);

          return String.format("Press button on the bridge '%s [%s]' within 30 seconds, refresh this page and try again if it doesn't work",
                               bridge.getIpAddress(), bridge.getMacAddress());
        }
      }
      return "Can't find bridge with ip: " + ip;
    }
  }
}
