package ru.org.amip.ambisync.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.philips.lighting.hue.listener.PHLightListener;
import com.philips.lighting.hue.sdk.PHHueSDK;
import com.philips.lighting.model.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.org.amip.ambisync.AmbiSync;
import ru.org.amip.ambisync.Controller;
import ru.org.amip.ambisync.Main;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.PrintWriter;
import java.util.*;

import static com.philips.lighting.model.PHLight.PHLightColorMode;
import static ru.org.amip.ambisync.Main.conf;

/**
 * Date: 16.03.2014 Time: 0:40
 *
 * @author serge
 */
public class Preset {
  private static Preset ourInstance = new Preset();

  private static final Logger logger = LoggerFactory.getLogger(Preset.class);

  public static final String PRESETS_FOLDER = "../conf/presets/";

  private volatile Timer delayTimer;

  private String lastPreset = null;

  private Map<String, PHLight> lightStates = new HashMap<>();


  public static Preset getInstance() {
    return ourInstance;
  }

  private Preset() {
    lastPreset = conf.getString("last.preset", null);

    //new Timer().schedule(new TimerTask() {
    //  @Override
    //  public void run() {
    //    lampStateWatchdog();
    //  }
    //}, 1000, 2000);
  }

  public String getLastPreset() {
    return lastPreset;
  }

  public ArrayList<String> getAll() {
    ArrayList<String> presets = new ArrayList<>();
    File[] files = new File(PRESETS_FOLDER).listFiles();

    if (files != null) {
      for (File file : files) {
        if (file.isFile() && file.getName().endsWith(".json")) {
          presets.add(file.getName().replace(".json", ""));
        }
      }
    }

    return presets;
  }

  public void load(final String name, long delay) {
    if (delayTimer != null) delayTimer.cancel();
    delayTimer = new Timer();
    delayTimer.schedule(new TimerTask() {
      @Override
      public void run() {
        load(name);
      }
    }, delay);
    logger.info("Preset '{}' will be applied in {}ms", name, delay);
  }

  public boolean load(String name) {
    if (AmbiSync.getInstance().isRunning()) {
      logger.info("Ambilight synchronization is in progress, preset '{}' is not applied", name);
      return false;
    }

    logger.info("Applying preset: '{}'", name);
    if (delayTimer != null) delayTimer.cancel();

    Gson gson = new Gson();
    try {
      final PHHueSDK phHueSDK = PHHueSDK.getInstance();
      final PHBridge bridge = phHueSDK.getSelectedBridge();
      if (bridge == null) return false;

      LampPreset[] lamps = gson.fromJson(new FileReader(PRESETS_FOLDER + name + ".json"), LampPreset[].class);

      for (final LampPreset lamp : lamps) {
        if (lamp == null) {
          logger.warn("Skipping invalid lamp in preset");
          continue;
        }

        logger.trace("{}", lamp);

        PHLightState lightState = new PHLightState();

        if (lamp.colormode != null) {
          PHLightColorMode cm = PHLightColorMode.COLORMODE_UNKNOWN;
          switch (lamp.colormode) {
            case "ct":
              cm = PHLightColorMode.COLORMODE_CT;
              break;
            case "hs":
              cm = PHLightColorMode.COLORMODE_HUE_SATURATION;
              break;
            case "xy":
              cm = PHLightColorMode.COLORMODE_XY;
              break;
          }
          lightState.setColorMode(cm);
        }

        lightState.setOn(lamp.on);

        if (lamp.bri != -1) lightState.setBrightness(lamp.bri);
        if (lamp.sat != -1) lightState.setSaturation(lamp.sat);
        if (lamp.hue != -1) lightState.setHue(lamp.hue);
        if (lamp.ct != -1) lightState.setCt(lamp.ct);

        if (lamp.xy != null) {
          lightState.setX(lamp.xy[0]);
          lightState.setY(lamp.xy[1]);
        }

        final PHLightListener listener = new PHLightListener() {
          @Override
          public void onReceivingLightDetails(PHLight light) {

          }

          @Override
          public void onReceivingLights(List<PHBridgeResource> list) {

          }

          @Override
          public void onSearchComplete() {

          }

          @Override
          public void onSuccess() {

          }

          @Override
          public void onError(int i, String s) {
            logger.warn("Lamp {} update failed: {}", lamp.id, s);
          }

          @Override
          public void onStateUpdate(Map<String, String> map, List<PHHueError> list) {

          }
        };

        bridge.updateLightState(lamp.id, lightState, listener);
        Controller.delay();
      }
      lastPreset = name;
      conf.setProperty("last.preset", lastPreset);
      Main.saveConfig();
      return true;
    } catch (FileNotFoundException e) {
      logger.error("Preset '{}' doesn't exist", name);
    }
    return false;
  }

  public boolean save(String name) {
    final PHHueSDK phHueSDK = PHHueSDK.getInstance();

    PHBridge bridge = phHueSDK.getSelectedBridge();
    if (bridge == null) return false;

    PHBridgeResourcesCache cache = bridge.getResourceCache();

    List<PHLight> allLights = cache.getAllLights();
    LampPreset[] lamps = new LampPreset[allLights.size()];


    int idx = 0;
    for (PHLight light : allLights) {
      final PHLightState state = new PHLightState(light.getLastKnownLightState());
      LampPreset lamp = new LampPreset();
      lamp.id = light.getIdentifier();

      if (state.getColorMode() == null) {
        logger.warn("Lamp {} color mode is null", lamp.id);
        lamp.colormode = null;
      } else {
        switch (state.getColorMode()) {
          case COLORMODE_UNKNOWN:
            lamp.colormode = null;
            break;
          case COLORMODE_NONE:
            lamp.colormode = null;
            break;
          case COLORMODE_CT:
            lamp.colormode = "ct";
            break;
          case COLORMODE_HUE_SATURATION:
            lamp.colormode = "hs";
            break;
          case COLORMODE_XY:
            lamp.colormode = "xy";
            break;
        }
      }

      lamp.on = state.isOn();
      lamp.bri = state.getBrightness();
      lamp.sat = state.getSaturation();
      lamp.hue = state.getHue();
      lamp.ct = state.getCt();
      lamp.xy = new float[2];
      lamp.xy[0] = state.getX();
      lamp.xy[1] = state.getY();
      lamps[idx] = lamp;
      idx++;
    }

    if (lamps.length > 0) {
      Gson gson = new GsonBuilder().setPrettyPrinting().create();
      String json = gson.toJson(lamps);
      logger.info("Saving '{}' preset", name);
      logger.trace(json);
      try {
        PrintWriter out = new PrintWriter(PRESETS_FOLDER + name + ".json");
        out.print(json);
        out.close();
        return true;
      } catch (FileNotFoundException e) {
        logger.error("Can't save preset '{}': {}", name, e.getMessage());
      }
    }
    return false;
  }

  private void lampStateWatchdog() {
    if (AmbiSync.getInstance().isRunning()) return;

    logger.warn("watchdog");

    final PHHueSDK phHueSDK = PHHueSDK.getInstance();
    PHBridge bridge = phHueSDK.getSelectedBridge();
    PHBridgeResourcesCache cache = bridge.getResourceCache();

    List<PHLight> allLights = cache.getAllLights();

    for (PHLight light : allLights) {
      final PHLight previousState = lightStates.get(light.getIdentifier());
      if (light.getLastKnownLightState().isReachable() && previousState != null && !previousState.getLastKnownLightState().isReachable()) {
        final String preset = Preset.getInstance().getLastPreset();
        if (preset != null) {
          logger.warn("Lamp {} was turned off and now is back on, restoring last preset '{}'", light.getIdentifier(), preset);
          Preset.getInstance().load(preset);
        }
      }

      lightStates.put(light.getIdentifier(), new PHLight(light));
      if (!light.getLastKnownLightState().isReachable()) {
        logger.warn("Lamp {} is no longer reachable", light.getIdentifier());
      }
    }
  }
}
