package ru.org.amip.ambisync.config;

import com.google.gson.Gson;
import com.google.gson.JsonIOException;
import com.google.gson.JsonSyntaxException;
import org.apache.commons.exec.*;
import org.apache.commons.exec.util.StringUtils;
import org.apache.http.client.fluent.Request;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * Date: 16.03.2014 Time: 0:40
 *
 * @author serge
 */
public class Command {
  private static final Logger logger = LoggerFactory.getLogger(Command.class);

  private static Command ourInstance = new Command();

  public static final String PRESETS_FOLDER = "../conf/cc-presets/";

  private HashMap<String, CommandPreset> presets = new HashMap<>();

  public static Command getInstance() {
    return ourInstance;
  }

  private Command() {
    loadPresets();
  }

  public HashMap<String, CommandPreset> getPresets() {
    return presets;
  }

  public boolean handleCommand(String trigger, Map<String, String> params) {
    if (!presets.containsKey(trigger)) {
      logger.warn("Command preset with '{}' trigger not found", trigger);
      return false;
    }
    final CommandPreset cmd = presets.get(trigger);
    switch (cmd.type) {
      case "exec": {
        handleExec(cmd, params);
        break;
      }
      case "http": {
        handleGet(cmd, params);
        break;
      }
      default:
        logger.warn("Unknown command type: {}", cmd.type);
        return false;
    }
    return true;
  }

  private void handleGet(CommandPreset cmd, Map<String, String> params) {
    try {
      final String request = StringUtils.stringSubstitution(cmd.command, params, true).toString();
      Request.Get(request)
             .connectTimeout(5000)
             .socketTimeout(5000)
             .execute();
      logger.info("'{}' http trigger handled ({})", cmd.trigger, cmd.command);
    } catch (IOException e) {
      logger.warn("HTTP request ({}) for trigger '{}' has failed", cmd.command, cmd.trigger, e);
    }
  }

  private void handleExec(CommandPreset cmd, Map<String, String> params) {
    CommandLine cmdLine = new CommandLine(cmd.command);
    cmdLine.setSubstitutionMap(params);

    if (cmd.args != null) {
      for (String arg : cmd.args) {
        cmdLine.addArgument(arg);
      }
    }

    DefaultExecuteResultHandler resultHandler = new DefaultExecuteResultHandler();

    ExecuteWatchdog watchdog = new ExecuteWatchdog(60 * 1000);
    Executor executor = new DefaultExecutor();
    executor.setExitValue(1);

    if (cmd.workdir == null) {
      executor.setWorkingDirectory(new File(cmd.command).getParentFile());
    } else {
      executor.setWorkingDirectory(new File(cmd.workdir));
    }

    executor.setWatchdog(watchdog);
    try {
      executor.execute(cmdLine, resultHandler);
      logger.info("'{}' exec trigger handled ({})", cmd.trigger, cmd.command);
    } catch (IOException e) {
      logger.warn("Command '{}' execution failed for '{}' trigger", cmd.command, cmd.trigger, e);
    }
  }

  public int loadPresets() {
    Gson gson = new Gson();
    presets.clear();
    File[] files = new File(PRESETS_FOLDER).listFiles();

    int total = 0;

    if (files != null) {
      for (File file : files) {
        if (file.isFile() && file.getName().endsWith(".json")) {
          try {
            CommandPreset cmd = gson.fromJson(new FileReader(PRESETS_FOLDER + file.getName()), CommandPreset.class);
            if (cmd.trigger != null && cmd.command != null) {
              if (presets.containsKey(cmd.trigger)) {
                logger.warn("Duplicate trigger {} detected in command preset {}", cmd.trigger, file.getName());
              }
              presets.put(cmd.trigger, cmd);
              total++;
              if (cmd.aliases != null) {
                for (String alias : cmd.aliases) {
                  if (presets.containsKey(alias)) {
                    logger.warn("Duplicate trigger alias {} detected in command preset {}", alias, file.getName());
                  }
                  presets.put(alias, cmd);
                }
              }
            } else {
              logger.warn("Preset in {} doesn't have a trigger or command defined", file.getName());
            }
          } catch (JsonSyntaxException | JsonIOException | FileNotFoundException e) {
            logger.error("Failed to load command preset {}", file.getName(), e);
          }
        }
      }
    }
    if (!presets.isEmpty()) {
      logger.info("Loaded {} command presets!", total);
    }
    return total;
  }
}
