package ru.org.amip.ambisync;

import com.philips.lighting.hue.sdk.PHAccessPoint;
import com.philips.lighting.hue.sdk.PHBridgeSearchManager;
import com.philips.lighting.hue.sdk.PHHueSDK;
import com.philips.lighting.hue.sdk.PHSDKListener;
import com.philips.lighting.model.PHBridge;
import com.philips.lighting.model.PHHueParsingError;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

import static ru.org.amip.ambisync.Main.conf;

public class Controller {
  private static final Logger logger = LoggerFactory.getLogger(Controller.class);

  private static Controller instance = new Controller();

  private PHHueSDK phHueSDK;
  private AmbiSync al;
  private PHBridge lastConnectedBridge;

  private volatile boolean connectionLost = false;

  private List<PHAccessPoint> bridges = new ArrayList<>();

  public static Controller getInstance() {
    return instance;
  }

  private volatile boolean isSyncEnabled = false;

  public void setSyncEnabled(boolean isSyncEnabled) {
    this.isSyncEnabled = isSyncEnabled;
  }

  private Controller() {
    this.phHueSDK = PHHueSDK.getInstance();
    phHueSDK.setAppName("HAmbiSync");
    phHueSDK.setDeviceName(System.getProperty("os.name"));
    al = AmbiSync.getInstance();
    isSyncEnabled = conf.getBoolean("sync.autostart");
  }

  public synchronized List<PHAccessPoint> getBridges() {
    return bridges;
  }

  public void findBridges() {
    logger.info("Searching for bridges, please wait...");
    PHBridgeSearchManager sm = (PHBridgeSearchManager) phHueSDK.getSDKService(PHHueSDK.SEARCH_BRIDGE);
    sm.search(true, true);
  }

  public void shutdown() {
    if (lastConnectedBridge != null) {
      phHueSDK.disableHeartbeat(lastConnectedBridge);
      phHueSDK.disconnect(lastConnectedBridge);
    }
    phHueSDK.destroySDK();
  }

  private long cacheUpdated;

  public synchronized long getCacheUpdated() {
    return cacheUpdated;
  }

  public synchronized void setCacheUpdated(long cacheUpdated) {
    this.cacheUpdated = cacheUpdated;
  }

  private PHSDKListener listener = new PHSDKListener() {

    @Override
    public void onAccessPointsFound(List<PHAccessPoint> accessPointsList) {
      logger.info("Found {} bridges", accessPointsList.size());
      String username = conf.getString("bridge.username");

      bridges.clear();
      PHAccessPoint autoConnect = null;
      for (PHAccessPoint point : accessPointsList) {
        bridges.add(point);
        logger.info("  > {} [{}]", point.getIpAddress(), point.getMacAddress());
        if ((accessPointsList.size() == 1 || point.getIpAddress().equalsIgnoreCase(conf.getString("bridge.ip"))) &&
            username != null && !username.isEmpty()) {
          autoConnect = point;
        }
      }

      if (autoConnect != null) {
        autoConnect.setUsername(username);
        phHueSDK.connect(autoConnect);
      }
    }

    @Override
    public void onCacheUpdated(List<Integer> list, PHBridge bridge) {
      logger.trace("Bridge cache updated");
      setCacheUpdated(System.currentTimeMillis());
      //lampStateWatchdog();
    }

    @Override
    public void onBridgeConnected(PHBridge bridge, String user) {
      logger.trace("Bridge connected");
      lastConnectedBridge = bridge;
      phHueSDK.setSelectedBridge(bridge);
      phHueSDK.enableHeartbeat(bridge, PHHueSDK.HB_INTERVAL);
      String lastIpAddress = bridge.getResourceCache().getBridgeConfiguration().getIpAddress();
      logger.info("Connected to bridge: {} [{}]", lastIpAddress, user);
      conf.setProperty("bridge.username", user);
      conf.setProperty("bridge.ip", lastIpAddress);
      Main.saveConfig();
      if (isSyncEnabled) al.startSync(true);
    }

    @Override
    public void onAuthenticationRequired(PHAccessPoint accessPoint) {
      logger.warn("Bridge '{}' requires authentication, press a button on this bridge within 30 seconds", accessPoint.getIpAddress());
      phHueSDK.startPushlinkAuthentication(accessPoint);
    }

    @Override
    public void onConnectionLost(PHAccessPoint arg0) {
      logger.warn("Bridge connection lost");
      connectionLost = true;
      al.stopSync(false);
    }

    @Override
    public void onParsingErrors(List<PHHueParsingError> list) {
      logger.warn("Parsing errors:");
      for (PHHueParsingError error : list) {
        logger.warn("{}", error.getMessage());
      }
    }

    @Override
    public void onConnectionResumed(PHBridge arg0) {
      logger.trace("Bridge connection resumed");
      if (connectionLost) {
        connectionLost = false;
        if (isSyncEnabled) al.startSync(true);
      }
    }

    @Override
    public void onError(int code, final String message) {
      logger.error("{} (code: {})", message, code);
    }
  };

  public PHSDKListener getListener() {
    return listener;
  }

  public boolean connectToLastKnownAccessPoint() {
    String username = conf.getString("bridge.username");
    String lastIpAddress = conf.getString("bridge.ip");

    if (username == null || lastIpAddress == null || username.isEmpty() || lastIpAddress.isEmpty()) {
      logger.error("Missing Last Username or Last IP.  Last known connection not found.");
      return false;
    }
    PHAccessPoint accessPoint = new PHAccessPoint();
    accessPoint.setIpAddress(lastIpAddress);
    accessPoint.setUsername(username);
    phHueSDK.connect(accessPoint);
    return true;
  }

  public static void delay() {
    final int delay = conf.getInt("bridge.update.delay");
    if (delay > 0) {
      try {
        Thread.sleep(delay);
      } catch (InterruptedException e) {
        e.printStackTrace();
      }
    }
  }

  public static void delay(long ms) {
    try {
      Thread.sleep(ms);
    } catch (InterruptedException e) {
      e.printStackTrace();
    }
  }
}
