/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.highlevel;

import amip.api.highlevel.exceptions.GeneralClientException;

/**
 * Interface to the player, allows to control the playback and get info about state and modes. For more advanced
 * functions and variables, consider using {@link Client#execute(String)} and {@link Client#evaluate(String)}.
 */
public class Player {
  private Client cl;

  public static final int STATUS_PLAYING = 1;
  public static final int STATUS_PAUSED = 3;
  public static final int STATUS_STOPPED = 0;

  Player() {
  }

  Player(Client cl) {
    this.cl = cl;
  }

  /**
   * Starts plating if player was paused or stopped, otherwise plays the song from the beginning.
   *
   * @throws GeneralClientException
   */
  public void play() throws GeneralClientException {
    cl.execute("control play");
  }

  /**
   * Plays song with zero based index.
   *
   * @param i song index, zero based.
   * @throws GeneralClientException
   */
  public void play(int i) throws GeneralClientException {
    i++;
    cl.execute("play " + i);
  }

  /**
   * Toggles pause.
   *
   * @throws GeneralClientException
   */
  public void pause() throws GeneralClientException {
    cl.execute("control pause");
  }

  /**
   * Stops the player.
   *
   * @throws GeneralClientException
   */
  public void stop() throws GeneralClientException {
    cl.execute("control stop");
  }

  /**
   * Moves to the next song in the playlist (if shuffle is enabled, then moves to the random song).
   *
   * @throws GeneralClientException
   */
  public void next() throws GeneralClientException {
    cl.execute("control >");
  }

  /**
   * Moves to the previous song in the playlist.
   *
   * @throws GeneralClientException
   */
  public void previous() throws GeneralClientException {
    cl.execute("control <");
  }

  /**
   * Sets player volume.
   *
   * @param volume 0-255
   * @throws GeneralClientException
   */
  public void setVolume(int volume) throws GeneralClientException {
    cl.execute("control vol " + volume);
  }

  public void volumeUp() throws GeneralClientException {
    cl.execute("control vup");
  }

  public void volumeDown() throws GeneralClientException {
    cl.execute("control vdwn");
  }

  /**
   * 5 seconds forward.
   *
   * @throws GeneralClientException
   */
  public void fastForward() throws GeneralClientException {
    cl.execute("control ff");
  }

  /**
   * Rewind 5 seconds.
   *
   * @throws GeneralClientException
   */
  public void rewind() throws GeneralClientException {
    cl.execute("control rew");
  }

  /**
   * Jumps to specified time from the beginning of the current song.
   *
   * @param time jump time in seconds.
   * @throws GeneralClientException
   */
  public void jump(int time) throws GeneralClientException {
    cl.execute("jumptotime " + time);
  }

  /**
   * Minimizes player window (Winamp only at the moment).
   *
   * @throws GeneralClientException
   */
  public void minimizeWindow() throws GeneralClientException {
    cl.execute("control mimimize");
  }

  /**
   * Restores player window (Winamp only at the moment).
   *
   * @throws GeneralClientException
   */
  public void restoreWindow() throws GeneralClientException {
    cl.execute("control restore");
  }

  /**
   * Allows to control shuffle mode.
   *
   * @param shuffle true to enable, false to disable.
   * @throws GeneralClientException
   */
  public void setShuffle(boolean shuffle) throws GeneralClientException {
    cl.execute("setshuffle " + (shuffle ? "on" : "off"));
  }

  /**
   * Returns current shuffle state.
   *
   * @return shuffle state.
   * @throws GeneralClientException
   */
  public boolean isShuffle() throws GeneralClientException {
    return "on".equalsIgnoreCase(cl.evaluate("var_shuffle"));
  }

  /**
   * Controls repeat mode.
   *
   * @param repeat true or false.
   * @throws GeneralClientException
   */
  public void setRepeat(boolean repeat) throws GeneralClientException {
    cl.execute("setrepeat " + (repeat ? "on" : "off"));
  }

  /**
   * Returns current repeat mode.
   *
   * @return true if enabled, false otherwise.
   * @throws GeneralClientException
   */
  public boolean isRepeat() throws GeneralClientException {
    return "on".equalsIgnoreCase(cl.evaluate("var_repeat"));
  }

  /**
   * Displays AMIP options (native dialog in case AMIP Configurator is not installed or runs AMIP Configurator if
   * CFG_JCONFIG in plugin.ini is set to 1).
   *
   * @throws GeneralClientException
   */
  public void showAMIPOptions() throws GeneralClientException {
    cl.execute("options");
  }

  /**
   * Opens AMIP Search/Jump dialog.
   *
   * @throws GeneralClientException
   */
  public void showAMIPSearch() throws GeneralClientException {
    cl.execute("search");
  }

  /**
   * Opens AMIP help.
   *
   * @throws GeneralClientException
   */
  public void showAMIPHelp() throws GeneralClientException {
    cl.execute("help");
  }

  /**
   * Gets position in the playlist, zero based.
   *
   * @return playlist position, zero based.
   * @throws GeneralClientException
   */
  public int getListPosition() throws GeneralClientException {
    int res = -1;
    try {
      res = Integer.parseInt(cl.evaluate("var_pos"));
    } catch (NumberFormatException e) {}
    if (res != -1) {
      res--;
    }
    return res;
  }

  /**
   * Gets current song length.
   *
   * @return song length in seconds.
   * @throws GeneralClientException
   */
  public int getSongLength() throws GeneralClientException {
    int res = -1;
    try {
      res = Integer.parseInt(cl.evaluate("var_sl"));
    } catch (NumberFormatException e) {}
    return res;
  }

  /**
   * Gets player version string.
   *
   * @return player version.
   * @throws GeneralClientException
   */
  public String getVersion() throws GeneralClientException {
    return cl.evaluate("var_ver");
  }

  /**
   * Gets player name string.
   *
   * @return player name.
   * @throws GeneralClientException
   */
  public String getName() throws GeneralClientException {
    return cl.evaluate("var_player");
  }

  /**
   * Gets AMIP version string.
   *
   * @return AMIP version.
   * @throws GeneralClientException
   */
  public String getAMIPVersion() throws GeneralClientException {
    return cl.evaluate("var_pv");
  }

  /**
   * Returns player status.
   *
   * @return STATUS_PLAYING (1), STATUS_PAUSED (3) or STATUS_STOPPED (0).
   * @throws GeneralClientException
   */
  public int getStatus() throws GeneralClientException {
    int res = 0;
    try {
      res = Integer.parseInt(cl.evaluate("var_stat"));
    } catch (NumberFormatException e) {}
    return res;
  }
}
