/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.highlevel.util;

import amip.api.highlevel.EventListenerManager;
import amip.api.highlevel.Server;
import amip.api.highlevel.exceptions.GeneralClientException;
import amip.api.wrapper.AMIPAPI;
import amip.api.wrapper.constants.Errors;

import javax.swing.event.EventListenerList;
import java.util.EventListener;

public class AEventListenerList extends EventListenerList {
  private int mode;

  public synchronized void addListener(Class c, EventListener l) throws GeneralClientException {
    super.add(c, l);
    int code = Utils.eventClassToCode(c);
    // listener for this event type has not been set yet
    if ((code & mode) != code) {
      mode += code;
      Server srv = Server.getInstance();
      EventListenerManager elm = srv.getEventListenersManager();
      int res =
        AMIPAPI.add_event_listener(srv.getExternalAddress(), srv.getPort(), elm.getTimeout(), mode, elm.getRetries());
      // adding listener has failed, set mode to the old value
      if(res != Errors.AC_ERR_NOERROR) mode -= code;
      Utils.errorToException(res);
    }
  }

  public synchronized void removeListener(Class c, EventListener l) throws GeneralClientException {
    super.remove(c, l);
    // no listeners set when mode is zero
    if (mode == 0) return;

    int code = Utils.eventClassToCode(c);
    // AMIP listener is still set for the event but the last eventlistener has been just removed
    if (((code & mode) == code) && super.getListenerCount(c) == 0) {
      mode -= code;
      Server srv = Server.getInstance();
      EventListenerManager elm = srv.getEventListenersManager();
      assert (mode >= 0);
      if (mode > 0) {
        // change the event listener mode
        int res =
          AMIPAPI.add_event_listener(srv.getExternalAddress(), srv.getPort(), elm.getTimeout(), mode, elm.getRetries());
        Utils.errorToException(res);
      } else if (mode == 0) {
        // last listener removed, remove the listener completely from AMIP
        int res = AMIPAPI.remove_event_listener(srv.getExternalAddress(), srv.getPort());
        Utils.errorToException(res);
      }
    }
  }

  public synchronized int getMode() {
    return mode;
  }

  public synchronized void setMode(int mode) {
    this.mode = mode;
  }
}
