/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.highlevel;

import amip.api.highlevel.exceptions.ServerStartupFailedException;
import amip.api.wrapper.AMIPAPI;

import java.io.IOException;
import java.net.*;
import java.util.ArrayList;
import java.util.Enumeration;

/**
 * Server class operates the server on your side. Server is required when you want to receive events and messages from
 * AMIP. Server opens a socket on the specified address and listens for AMIP connections. This class also provides
 * access to the {@link EventListenerManager}. Only one instance of Server can be created. To get the Server instance,
 * use {@link #getInstance} method. Once you have it, you need to start the server. Server must be stopped when you exit
 * your application.
 */
public class Server {
  private static Server ourInstance = null;
  private EventListenerManager ourEventListenerManager = null;

  private String bindAddress = "127.0.0.1";
  private String externalAddress = null;

  private boolean started = false;

  private int port = 60334;

  private static final int PORT_MIN = 60334;
  private static final int PORT_MAX = 60666;

  private Server() {
  }

  /**
   * Starts the server on the specified bindAddress and port. bindAddress must be a valid network address, use 127.0.0.1
   * to start server on the localhost. If you expect connections from the AMIP installed on another machine, you have to
   * either start server on the external address which is visible by that machine or start it on all the addresses your
   * machine has by specifying "0.0.0.0" as a bindAddress. If you start server on all the addresses, you must specify
   * the external address using the {@link #setExternalAddress} method, this address is passed to AMIP so that it knows
   * where it should connect.
   *
   * @param bindAddress valid IP address of your machine, or 127.0.0.1 if AMIP is on the same machine, or 0.0.0.0 if you
   *                    want to listen on all addresses.
   * @param port        port to start the server on, use one of the ports which is free or get one using the
   *                    findFreePort method.
   * @throws ServerStartupFailedException if server was not started, usually it happens if this port is already taken by
   *                                      another application or specified IP address is invalid.
   */
  public void start(String bindAddress, int port) throws ServerStartupFailedException {
    this.bindAddress = bindAddress;
    this.port = port;
    start();
  }

  /**
   * Starts the server on 127.0.0.1:60334 or another address and port previously specified by setBindAddress and setPort
   * methods.
   *
   * @throws ServerStartupFailedException if the server cannot be started (usually when port is already in use or
   *                                      specified interface doesn't exist).
   */
  public void start() throws ServerStartupFailedException {
    int res = AMIPAPI.init_server(bindAddress, port);
    if (res == 0) throw new ServerStartupFailedException();
    started = true;
  }

  /** Stops the server, you must stop it before your application is terminated. */
  public void stop() {
    if (started) AMIPAPI.stop_server();
    started = false;
  }

  private static synchronized void createInstance() {
    if (ourInstance == null) {
      ourInstance = new Server();
    }
  }

  /**
   * Gets the {@link Server} instance, this is the only way to get the Server object.
   *
   * @return instance of the Server.
   */
  public static synchronized Server getInstance() {
    if (ourInstance == null) createInstance();
    return ourInstance;
  }

  /**
   * Provides access to {@link EventListenerManager} useful when you need to receive events from AMIP.
   *
   * @return {@link EventListenerManager} instance.
   */
  public EventListenerManager getEventListenersManager() {
    if (ourEventListenerManager == null) ourEventListenerManager = new EventListenerManager();
    return ourEventListenerManager;
  }

  public String getBindAddress() {
    return bindAddress;
  }

  public void setBindAddress(String bindAddress) {
    this.bindAddress = bindAddress;
  }

  public int getPort() {
    return port;
  }

  public void setPort(int port) {
    this.port = port;
  }

  public boolean isStarted() {
    return started;
  }

  /**
   * This method returns an external address of the server, which equals to bindAddress by default or null if
   * bindAddress is "0.0.0.0" and externalAddress was not explicitly set.
   *
   * @return external server IP address or null if server is listening on all addresses and externalAddress was not
   *         explicitly set.
   */
  public String getExternalAddress() {
    if (externalAddress == null) {
      if (bindAddress.compareTo("0.0.0.0") == 0)
        return null;
      else
        return bindAddress;
    } else {
      return externalAddress;
    }
  }

  /**
   * You only need to set it if you start your server on all the addresses available on your machine.
   *
   * @param externalAddress machine external address.
   */
  public void setExternalAddress(String externalAddress) {
    this.externalAddress = externalAddress;
  }

  /**
   * Helper method to find a free port for the server. The port is considered free only if it's possible to bind to this
   * port using each of the available network interfaces.
   *
   * @return free port or -1 if no free port was found within certain limits.
   * @noinspection SocketOpenedButNotSafelyClosed,EmptyCatchBlock
   */
  public static int findFreePort() {
    InetAddress[] interfaces = getNetworkInterfaces();
    return findFreePort(interfaces);
  }

  /**
   * The same as {@link #findFreePort}, but looks for free port only on the local address.
   *
   * @return free port or -1 if not found
   */
  public static int findFreeLocalPort() {
    try {
      return findFreePort(new InetAddress[]{InetAddress.getByName("127.0.0.1")});
    } catch (UnknownHostException e) {
      return -1;
    }
  }

  /**
   * Searches for the free port in range on the specified interface.
   *
   * @param iface network interface IP address in a string form
   * @return free port or -1 if not found
   */
  public static int findFreePort(String iface) {
    if (iface.compareTo("0.0.0.0") == 0) return findFreePort();
    try {
      return findFreePort(new InetAddress[]{InetAddress.getByName(iface)});
    } catch (UnknownHostException e) {
      return -1;
    }
  }

  /**
   * Finds random free port by creating a server socket with no port specified and returning its port
   *
   * @return random free port to bind on
   */
  public static int findRandomPort() {
    ServerSocket socket = null;
    try {
      socket = new ServerSocket(0);
      return socket.getLocalPort();
    } catch (IOException e) {
    } finally {
      if (socket != null) {
        try {
          socket.close();
        } catch (IOException ignore) { }
      }
    }
    return -1;
  }

  private static int findFreePort(InetAddress[] interfaces) {
    for (int i = PORT_MIN; i < PORT_MAX; i++) {
      boolean failed = false;
      for (int j = 0; j < interfaces.length; j++) {
        InetAddress ia = interfaces[j];
        ServerSocket socket = null;
        try {
          socket = new ServerSocket(i, 50, ia);
        } catch (IOException e) {
          failed = true;
          break;
        } finally {
          if (socket != null) try {
            socket.close();
          } catch (IOException e) { }
        }
      }
      if (!failed) return i;
    }
    return -1;
  }

  private static InetAddress[] getNetworkInterfaces() {
    ArrayList ifaces = new ArrayList();
    try {
      Enumeration networkInterfaces = NetworkInterface.getNetworkInterfaces();
      while (networkInterfaces.hasMoreElements()) {
        NetworkInterface netif = (NetworkInterface) networkInterfaces.nextElement();
        Enumeration inetAddresses = netif.getInetAddresses();
        while (inetAddresses.hasMoreElements()) {
          InetAddress inetAddress = (InetAddress) inetAddresses.nextElement();
          if (inetAddress instanceof Inet4Address) ifaces.add(inetAddress);
        }
      }
    } catch (SocketException e) { }
    return (InetAddress[]) ifaces.toArray(new InetAddress[ifaces.size()]);
  }
}
