/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.highlevel;

import amip.api.highlevel.exceptions.GeneralClientException;
import amip.api.highlevel.listeners.PlaylistChangeEventListener;
import amip.api.highlevel.listeners.PlaylistReadyEventListener;
import amip.api.highlevel.listeners.SyncCompleteEventListener;
import amip.api.highlevel.util.Semaphore;
import amip.api.highlevel.util.Utils;
import amip.api.wrapper.AMIPAPI;

import javax.swing.event.EventListenerList;
import java.util.ArrayList;

/**
 * This class provides an easy read-only access to the playlist. You can get the whole playlist as an {@link ArrayList}.
 * There are 2 ways to get the playlist, the async way and sync. If you are connecting to remote AMIP over a slow
 * connection or playlist is very large or you want your GUI to stay responsive, consider using async way.
 * <p/>
 * The most simple way to get the playlist is the syncAndGetCopy method. Playlist instance itself is returned by {@link
 * Client#getPlaylist()}. {@link Client} and {@link Server} must be already initialized before making any calls to the
 * {@link Playlist} methods.
 */
public class Playlist {
  private final Object sync = new Object();
  private EventListenerList listenerList = new EventListenerList();

  private ArrayList plist;

  private boolean autoSynchronize = false;
  private boolean shouldProcessReadyEvent = false;

  private PlaylistChangeEventListener ourChangeListener;
  private PlaylistReadyEventListener ourReadyListener;

  Playlist() {
  }

  /**
   * Adds listener which is notified when playlist synchronization with player is complete.
   *
   * @param l listener.
   */
  public void addSyncCompleteListener(SyncCompleteEventListener l) {
    synchronized (sync) {
      listenerList.add(SyncCompleteEventListener.class, l);
    }
  }

  /**
   * Removes previously added listener.
   *
   * @param l listener.
   */
  public void removeSyncCompleteListener(SyncCompleteEventListener l) {
    synchronized (sync) {
      listenerList.remove(SyncCompleteEventListener.class, l);
    }
  }

  private void fireSyncComplete() {
    synchronized (sync) {
      Object[] listeners = listenerList.getListenerList();
      for (int i = listeners.length - 2; i >= 0; i -= 2) {
        ((SyncCompleteEventListener) listeners[i + 1]).syncComplete();
      }
    }
  }

  public boolean isAutoSynchronize() {
    return autoSynchronize;
  }

  /**
   * Returns the number of entries in the playlist.
   *
   * @return the size of the playlist, 0 is returned in case playlist is not synced.
   */
  public int getSize() {
    synchronized (sync) {
      return plist == null ? 0 : plist.size();
    }
  }

  /**
   * Gets the title from playlist with the specified index.
   *
   * @param index the index of title.
   * @return playlist title or <code>null</code> if index is not within bounds or title is not available.
   */
  public String getTitle(int index) {
    synchronized (sync) {
      return (String) (getSize() >= index + 1 ? null : plist.get(index));
    }
  }

  /**
   * Once playlist is synced with the {@link #sync()} method you can get its copy using this method, it's recommended to
   * call this method in your {@link SyncCompleteEventListener} class to be sure that the list is in sync.
   *
   * @return a copy of the playlist as a Java {@link ArrayList}, returns empty array list if the list is not synced.
   */
  public ArrayList getCopy() {
    synchronized (sync) {
      return (ArrayList) (plist == null ? new ArrayList() : plist.clone());
    }
  }

  /**
   * The easiest way to get the playlist, it automatically invokes sync with the player, waits until sync is complete
   * and returns a copy like {@link #getCopy()} method.
   *
   * @return a copy of the playlist as a Java {@link ArrayList}, returns empty array list if the list is not synced.
   * @throws GeneralClientException
   * @throws InterruptedException
   */
  public ArrayList syncAndGetCopy() throws GeneralClientException, InterruptedException {
    final Semaphore s = new Semaphore();
    SyncCompleteEventListener scl = new SyncCompleteEventListener() {
      public void syncComplete() {
        s.release();
      }
    };
    addSyncCompleteListener(scl);
    sync();
    s.acquire();
    removeSyncCompleteListener(scl);
    synchronized (sync) {
      return getCopy();
    }
  }

  /**
   * Checks if AMIP's playlist is in sync with the players playlist.
   *
   * @return true if the list is in sync, false otherwise.
   */
  public boolean isInSync() {
    return "1".equals(AMIPAPI.eval("var_listinsync"));
  }

  /**
   * Enables the automatic sync mode. In this mode when the playlist is changed in player, AMIP will automatically sync
   * with it. It just adds the {@link PlaylistChangeEventListener} and calls {@link #sync()} method in it.
   *
   * @param autoSynchronize true/false.
   * @throws GeneralClientException
   */
  public void setAutoSynchronize(boolean autoSynchronize) throws GeneralClientException {
    EventListenerManager elm = Server.getInstance().getEventListenersManager();
    if (!this.autoSynchronize && autoSynchronize) {
      ourChangeListener = new PlaylistChangeEventListener() {
        public void eventReceived() {
          try {
            sync();
          } catch (GeneralClientException e) {
            // hide
          }
        }
      };
    } else if (this.autoSynchronize && !autoSynchronize) {
      elm.removeEventListener(ourChangeListener);
    }

    if (autoSynchronize && ourChangeListener != null) {
      // re-add in case player has restarted and listeners were removed
      elm.removeEventListener(ourChangeListener);
      elm.addEventListener(ourChangeListener);
    }

    this.autoSynchronize = autoSynchronize;
  }

  /**
   * Synchronizes playlist with player and AMIP, if AMIP's and player's lists are not in sync, it automatically invokes
   * reindexing in AMIP. Once sync is complete, {@link SyncCompleteEventListener} listeners are notified about it.
   *
   * @throws GeneralClientException
   */
  public void sync() throws GeneralClientException {
    shouldProcessReadyEvent = true;
    EventListenerManager elm = Server.getInstance().getEventListenersManager();
    if (ourReadyListener == null) {
      ourReadyListener = new PlaylistReadyEventListener() {
        public void eventReceived() {
          if (shouldProcessReadyEvent) {
            shouldProcessReadyEvent = false;
            processReadyEvent();
          }
        }
      };
    }
    // re-add in case player has restarted and listeners were removed
    elm.removeEventListener(ourReadyListener);
    elm.addEventListener(ourReadyListener);
    if (!isInSync()) {
      Client.getInstance().execute("reindexq");
    } else {
      processReadyEvent();
    }
  }

  private void processReadyEvent() {
    synchronized (sync) {
      try {
        int res = AMIPAPI.get_pl();
        Utils.errorToException(res);
      } catch (GeneralClientException e) {
        // ignore
      }

      if (plist != null) plist.clear();

      int plsize = AMIPAPI.get_plsize();

      if (plist == null) {
        plist = new ArrayList(plsize);
      } else {
        plist.ensureCapacity(plsize);
      }

      for (int i = 0; i < plsize; i++) {
        plist.add(i, AMIPAPI.get_title(i));
      }

      fireSyncComplete();
    }
  }
}
