/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.highlevel;

import amip.api.highlevel.exceptions.GeneralClientException;
import amip.api.highlevel.listeners.AEventListener;
import amip.api.highlevel.listeners.MessageEventListener;
import amip.api.highlevel.listeners.ChangeEventListener;
import amip.api.highlevel.util.AEventListenerList;
import amip.api.highlevel.util.Utils;
import amip.api.wrapper.AMIPAPI;
import amip.api.wrapper.callbacks.EventCallback;
import amip.api.wrapper.callbacks.MessageCallback;

import java.util.EventListener;

/**
 * This class is used for operations with event listeners. You can easily configure listeners for the AMIP events you
 * are interested in and be notified when they occur. For example, you can add {@link
 * ChangeEventListener} to know when song changes in the player.
 * <p/>
 * The instance of this class can be obtained from {@link Server} using {@link Server#getEventListenersManager()}
 * method.
 */
public class EventListenerManager {
  private AEventListenerList listenerList = new AEventListenerList();

  private int timeout = 5000;
  private int retries = 5;

  EventListenerManager() {
  }

  /**
   * May be useful to know if there are any listeners set.
   *
   * @return the number of listeners.
   */
  public int getListenerCount() {
    return listenerList.getListenerCount();
  }

  /**
   * Adds an event listener.
   *
   * @param l the listener to add.
   * @throws GeneralClientException
   */
  public void addEventListener(AEventListener l) throws GeneralClientException {
    Class aClass = l.getClass().getInterfaces()[0];
    if (AMIPAPI.evtCallback == null) {
      AMIPAPI.setEvtCallback(new EventCallback() {
        public void process(int mode) {
          fireEvent(mode);
        }
      });
    }
    listenerList.addListener(aClass, l);
  }

  /**
   * Removes previously added event listener.
   *
   * @param l the listener to remove.
   * @throws GeneralClientException
   */
  public void removeEventListener(AEventListener l) throws GeneralClientException {
    listenerList.removeListener(l.getClass(), l);
  }

  /**
   * Adds listener for messages from AMIP.
   *
   * @param l the listener to add.
   * @throws GeneralClientException
   */
  public void addMessageEventListener(MessageEventListener l) throws GeneralClientException {
    if (AMIPAPI.msgCallback == null) {
      AMIPAPI.setMsgCallback(new MessageCallback() {
        public void process(String message) {
          fireMessageEvent(message);
        }
      });
    }
    listenerList.addListener(MessageEventListener.class, l);
  }

  /**
   * Removes message listener.
   *
   * @param l the listener to remove.
   * @throws GeneralClientException
   */
  public void removeMessageEventListener(MessageEventListener l) throws GeneralClientException {
    listenerList.removeListener(MessageEventListener.class, l);
  }

  /**
   * Removes all the event listeners. If you were listening for any events, either directly, or indirectly while working
   * with the {@link Playlist}, before exiting your application you need to call this method. You have to call it
   * before the uninit method of the {@link Client}.
   *
   * @throws GeneralClientException
   */
  public void removeAllEventListeners() throws GeneralClientException {
    if (getListenerCount() > 0) {
      int res =
        AMIPAPI.remove_event_listener(Server.getInstance().getExternalAddress(), Server.getInstance().getPort());
      Object[] listeners = listenerList.getListenerList();
      for (int i = listeners.length - 2; i >= 0; i -= 2) {
        //noinspection unchecked
        listenerList.remove((Class) listeners[i], (EventListener) listeners[i + 1]);
      }
      // restore mode to 0
      listenerList.setMode(0);
      assert (listenerList.getListenerCount() == 0);
      Utils.errorToException(res);
    }
  }


  /**
   * Simulates message event received from AMIP.
   *
   * @param msg a message string.
   */
  public void fireMessageEvent(String msg) {
    Object[] listeners = listenerList.getListenerList();
    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == MessageEventListener.class) {
        ((MessageEventListener) listeners[i + 1]).messageReceived(msg);
      }
    }
  }

  /**
   * Simulates event received from AMIP.
   *
   * @param code event code.
   */
  public void fireEvent(int code) {
    Class eventClass = Utils.eventCodeToClass(code);
    fireEvent(eventClass);
  }

  /**
   * Simulates event received from AMIP.
   *
   * @param eventClass class of the event to fire.
   */
  public void fireEvent(Class eventClass) {
    if (eventClass == null) return;
    Object[] listeners = listenerList.getListenerList();

    for (int i = listeners.length - 2; i >= 0; i -= 2) {
      if (listeners[i] == eventClass) {
        ((AEventListener) listeners[i + 1]).eventReceived();
      }
    }
  }

  public int getTimeout() {
    return timeout;
  }

  /**
   * Sets timeout for AMIP plugin while trying to connect to your {@link Server}.
   *
   * @param timeout timeout in milliseconds.
   */
  public void setTimeout(int timeout) {
    this.timeout = timeout;
  }

  public int getRetries() {
    return retries;
  }

  /**
   * Sets number of retries before the listener is automatically removed from AMIP.
   *
   * @param retries if AMIP can't notify you about the event this number of times, it will remove the listener and stop
   *                trying.
   */
  public void setRetries(int retries) {
    this.retries = retries;
  }
}
