/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.highlevel;

import amip.api.highlevel.exceptions.GeneralClientException;
import amip.api.highlevel.listeners.MessageEventListener;
import amip.api.highlevel.util.Semaphore;

import java.util.Map;
import java.util.TreeMap;

/** Provides easy access to AMIP configuration. */
public class Config {
  // This keyword is sent when dump is finished
  public static final String DUMP_END = "=DUMP_END=";
  // Config option for output command
  public static final String OUTPUT_COMMAND = "cmdstr";
  // Config option to disable transports
  public static final String DISABLE_TRANSPORTS = "disabletransports";

  private Client cl;

  Config(Client cl) {
    this.cl = cl;
  }

  /**
   * Sets AMIP config option.
   *
   * @param option option name.
   * @param value  option value.
   * @throws GeneralClientException
   */
  public synchronized void setOption(String option, String value) throws GeneralClientException {
    cl.execute("setvar " + option + " " + value);
  }

  /**
   * Gets AMIP config option value.
   *
   * @param option option to get.
   * @return value of the option.
   * @throws GeneralClientException
   */
  public synchronized String getOption(String option) throws GeneralClientException {
    return cl.evaluate("cfg_" + option);
  }

  /**
   * Saves config. Note, that you can change many options without the need to call this. If you've changed something
   * that requires config reloading, then call this method. Once config is saved, AMIP will automatically reload it and
   * apply all the settings.
   *
   * @throws GeneralClientException
   */
  public synchronized void save() throws GeneralClientException {
    cl.execute("config save");
  }

  /**
   * Loads config. If you load it without saving, all changes will be lost. Don't load config explicitly after saving it
   * with {@link #save()} method, AMIP loads it automatically when timestamp changes.
   *
   * @throws GeneralClientException
   */
  public synchronized void load() throws GeneralClientException {
    cl.execute("config load");
  }

  /**
   * Uses cfgdump API command to dump all the config options and values. May take some time to complete on slow
   * connections, don't call often.
   * <p/>
   * Warning: may hang if used with old AMIP 2.53 beta version that doesn't print =DUMP_END= when dump is finished.
   *
   * @return Map with sorted config keys and values.
   * @throws GeneralClientException
   * @throws InterruptedException
   */
  public synchronized Map getDump() throws GeneralClientException, InterruptedException {
    final Map cachedConfig = new TreeMap();
    final Semaphore s = new Semaphore();

    EventListenerManager manager = Server.getInstance().getEventListenersManager();

    MessageEventListener mel = new MessageEventListener() {
      public void messageReceived(String msg) {
        // sample format: >tbcolor:CFG_TBCOLOR:'000000'
        if (msg.startsWith(">")) {
          msg = msg.substring(1);
          // special message is sent when dump ends
          if (msg.equalsIgnoreCase(DUMP_END)) {
            // release semaphore
            s.release();
          } else {
            int key_end = msg.indexOf(':');
            if (key_end == -1) return;
            int value_start = msg.indexOf(':', key_end + 1);
            if (value_start == -1) return;

            String key = msg.substring(0, key_end);
            String val = msg.substring(value_start + 2, msg.length() - 1);

            cachedConfig.put(key, val);
          }
        }
      }
    };

    Config config = cl.getConfig();
    String oldoutput = config.getOption(OUTPUT_COMMAND);
    String oldtransports = config.getOption(DISABLE_TRANSPORTS);

    config.setOption(OUTPUT_COMMAND, "");
    config.setOption(DISABLE_TRANSPORTS, "1");

    manager.addMessageEventListener(mel);

    // dump config and wait until dump is finished
    cl.execute("cfgdump");
    s.acquire();

    // restore status output string and remove listener
    config.setOption(OUTPUT_COMMAND, oldoutput == null ? "" : oldoutput);
    config.setOption(DISABLE_TRANSPORTS, oldtransports);
    manager.removeMessageEventListener(mel);

    return cachedConfig;
  }
}
