/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.highlevel;

import amip.api.highlevel.exceptions.ClientInitializationFailedException;
import amip.api.highlevel.exceptions.GeneralClientException;
import amip.api.highlevel.util.Utils;
import amip.api.wrapper.AMIPAPI;

/**
 * This class provides access to AMIP itself and also Playlist and Player objects. You must init it before any other
 * communication with AMIP using the init method.
 */
public class Client {
  private static Client ourInstance;
  private Playlist plist;
  private Player player;
  private Config config;

  private String host = "127.0.0.1";
  private int port = 60333;
  private int timeout = 5000;
  private int dsec = 5;
  private int dcount = 1;
  private boolean initialized = false;

  private Client() {
  }

  private static synchronized void createInstance() {
    if (ourInstance == null) {
      ourInstance = new Client();
    }
  }

  /**
   * Use this method to get the {@link Client} object.
   *
   * @return Client instance.
   */
  public static synchronized Client getInstance() {
    if (ourInstance == null) createInstance();
    return ourInstance;
  }

  /**
   * Initializes client.
   *
   * @param host    specifies the IP address of the machine where AMIP plugin is installed and running, may be 127.0.0.1
   *                in case AMIP is on the same machine.
   * @param port    AMIP plugin port, default is 60333.
   * @param timeout timeout in milliseconds for all the operations, on localhost 1000-5000 is a good option, on slow and
   *                unstable connections you may want to use higher values.
   * @param dsec    amount of seconds to block any connections for if previous dcount attempts failed.
   * @param dcount  threshold of failed operations, if reached, all further connections will be suspended for dsec
   *                seconds. dsec and dcount arguments are to prevent lockups in case AMIP can't be reached.
   * @throws ClientInitializationFailedException
   *          if client cannot be initialized with the specified parameters.
   */
  public void init(String host, int port, int timeout, int dsec, int dcount) throws ClientInitializationFailedException {
    this.host = host;
    this.port = port;
    this.timeout = timeout;
    this.dsec = dsec;
    this.dcount = dcount;
    init();
  }

  /**
   * Simpe init method, uses defaults for timeout (5000), dsec (5) and dcount (1) parameters.
   *
   * @param host AMIP host.
   * @param port AMIP port.
   * @throws ClientInitializationFailedException
   *
   */
  public void init(String host, int port) throws ClientInitializationFailedException {
    this.host = host;
    this.port = port;
    init();
  }

  /**
   * You may use this method if AMIP is installed on the same machine and is running on the default port (60333). This
   * method initializes client for the 127.0.0.1:60333.
   *
   * @throws ClientInitializationFailedException
   *
   */
  public void init() throws ClientInitializationFailedException {
    int res = AMIPAPI.init_client(host, port, timeout, dsec, dcount);
    if (res == 0) throw new ClientInitializationFailedException();
    initialized = true;
  }

  /** Uninit client before your application is terminated. */
  public void uninit() {
    if (initialized) AMIPAPI.uninit_client();
    initialized = false;
  }

  /**
   * Evaluates AMIP variable and returns the result of evaluation as string.
   *
   * @param variable AMIP API command/variable to evaluate, for example 'var_name' or 'cfg_enabled'.
   * @return the result of evaluation.
   * @throws GeneralClientException
   */
  public String evaluate(String variable) throws GeneralClientException {
    String res = AMIPAPI.eval(variable);
    Utils.errorToException(AMIPAPI.get_last_error());
    return res;
  }

  /**
   * Executes AMIP API command.
   *
   * @param command API command, such as 'control pause'.
   * @throws GeneralClientException
   */
  public void execute(String command) throws GeneralClientException {
    int res = AMIPAPI.exec(command);
    Utils.errorToException(res);
  }

  /**
   * Formats the string with AMIP variables or functions.
   *
   * @param spec the string which may contain AMIP variables and functions, for example "$cf(%1 - %2)".
   * @return result the string with information provided by AMIP and formatted according to spec.
   * @throws GeneralClientException
   */
  public String format(String spec) throws GeneralClientException {
    String res = AMIPAPI.format(spec);
    Utils.errorToException(AMIPAPI.get_last_error());
    return res;
  }

  /**
   * Same as {@link #format(String spec)}, but uses AMIP preset with the specified number and formats the result
   * according to this preset.
   *
   * @param preset AMIP preset to format song info according to (can take value from 1 to 5).
   * @return the string with information provided by AMIP and formatted according to specified preset.
   * @throws GeneralClientException
   */
  public String format(int preset) throws GeneralClientException {
    String res = AMIPAPI.eval("formatpreset " + preset);
    Utils.errorToException(AMIPAPI.get_last_error());
    return res;
  }

  /**
   * Pings AMIP plugin, useful to know if it's running or not. Note that it returns false if player with installed AMIP
   * is not running on the host:port client was initialized for or in case AMIP Server/API is disabled.
   *
   * @return true if AMIP is running on the specified port and host, false otherwise.
   */
  public boolean pingServer() {
    return AMIPAPI.pingServer(host, port, timeout);
  }

  public int getDcount() {
    return dcount;
  }

  public void setDcount(int dcount) {
    this.dcount = dcount;
  }

  public int getDsec() {
    return dsec;
  }

  public void setDsec(int dsec) {
    this.dsec = dsec;
  }

  public int getTimeout() {
    return timeout;
  }

  public void setTimeout(int timeout) {
    this.timeout = timeout;
  }

  public int getPort() {
    return port;
  }

  public void setPort(int port) {
    this.port = port;
  }

  public String getHost() {
    return host;
  }

  public void setHost(String host) {
    this.host = host;
  }

  public boolean isInitialized() {
    return initialized;
  }

  /**
   * Provides access to the {@link Playlist} object for operations with player's playlist.
   *
   * @return Playlist instance.
   */
  public Playlist getPlaylist() {
    if (plist == null) {
      plist = new Playlist();
    }
    return plist;
  }

  /**
   * Returns {@link Player} instance to control your player.
   *
   * @return Player instance.
   */
  public Player getPlayer() {
    if (player == null) {
      player = new Player(this);
    }
    return player;
  }

  public Config getConfig() {
    if (config == null) {
      config = new Config(this);
    }
    return config;
  }
}
