/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.examples.highlevel;

import amip.api.examples.util.Helpers;
import amip.api.highlevel.Client;
import amip.api.highlevel.EventListenerManager;
import amip.api.highlevel.Server;
import amip.api.highlevel.listeners.ExitEventListener;
import amip.api.highlevel.listeners.MessageEventListener;
import amip.api.highlevel.listeners.PauseEventListener;
import amip.api.highlevel.listeners.PlayEventListener;

/**
 * EventWatcher shows how to register listeners for the events you are interested in and handle the received events.
 */
public class EventWatcher {
  public static void main(String[] args) {
    // we need both client and server, client is for controlling AMIP and setting listeners,
    // server is for receiving events from AMIP via callbacks
    Client c = Client.getInstance();
    Server s = Server.getInstance();

    try {
      c.init();
      s.start();

      EventListenerManager elm = s.getEventListenersManager();

      elm.addEventListener(new ExitEventListener() {
        public void eventReceived() {
          System.out.println("Player exiting");
        }
      });

      elm.addEventListener(new PauseEventListener() {
        public void eventReceived() {
          System.out.println("Player paused");
        }
      });

      elm.addEventListener(new PlayEventListener() {
        public void eventReceived() {
          System.out.println("Player unpaused");
        }
      });

      elm.addMessageEventListener(new MessageEventListener() {
        public void messageReceived(String msg) {
          System.out.println("Message received: " + msg);
        }
      });

      System.out.println("Listening for events... Press Ctrl+C to exit...");

      // since it's a console application we need a way to shutdown gracefully when Ctrl+C is pressed
      // Thread is locked here until Ctrl+C is pressed, but events still can be received since they are fired
      // in different thread
      Helpers.setupTerminationHandlerAndWait(Thread.currentThread());

      // remove all the listeners set above, always do it before exiting the application and client uninit() call
      elm.removeAllEventListeners();

    } catch (Exception e) {
      e.printStackTrace();
    } finally {
      s.stop();
      c.uninit();
    }
  }
}
