/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.examples.highlevel;

import amip.api.highlevel.Client;
import amip.api.highlevel.Config;
import amip.api.highlevel.EventListenerManager;
import amip.api.highlevel.Server;
import amip.api.highlevel.exceptions.ClientInitializationFailedException;
import amip.api.highlevel.exceptions.GeneralClientException;
import amip.api.highlevel.exceptions.ServerStartupFailedException;
import amip.api.highlevel.listeners.MessageEventListener;
import amip.api.highlevel.util.Semaphore;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * Dumps all the AMIP variables. Illustrates some internal features of AMIP. This sample may hang if used with the old
 * AMIP version that doesn't print =DUMP_END=.
 */
public class DumpVariables {
  private static final Semaphore ourSem = new Semaphore();
  private static Map ourMap = new TreeMap();

  public static void main(String[] args) throws ClientInitializationFailedException, ServerStartupFailedException, GeneralClientException, InterruptedException {
    final Client c = Client.getInstance();
    Server s = Server.getInstance();

    c.init();
    s.start();

    EventListenerManager manager = s.getEventListenersManager();
    manager.addMessageEventListener(new MessageEventListener() {
      public void messageReceived(String msg) {
        parseMessage(msg);
      }
    });

    Config config = c.getConfig();
    // save old output string
    String oldoutput = config.getOption(Config.OUTPUT_COMMAND);
    System.out.println("oldoutput = " + oldoutput);
    String oldtransports = config.getOption(Config.DISABLE_TRANSPORTS);
    // set empty str
    config.setOption(Config.OUTPUT_COMMAND, "");
    // disable transports
    config.setOption(Config.DISABLE_TRANSPORTS, "1");

    c.execute("dump");
    ourSem.acquire();

    // at this place we should have ourMap with AMIP key=value pairs
    Set set = ourMap.keySet();
    for (Iterator iterator = set.iterator(); iterator.hasNext();) {
      String key = (String) iterator.next();
      String val = (String) ourMap.get(key);
      if (val.length() == 0) val = "(empty)";
      val = val.replaceAll("\r\n", "\\\\n");
      System.out.println(key + " = " + val);
    }

    config.setOption(Config.OUTPUT_COMMAND, oldoutput == null ? "" : oldoutput);
    config.setOption(Config.DISABLE_TRANSPORTS, oldtransports);
    manager.removeAllEventListeners();
    c.uninit();
    s.stop();
  }

  private static void parseMessage(String msg) {
    if (msg.startsWith(">")) {
      msg = msg.substring(1);
      // special message is sent when dump ends
      if (msg.equalsIgnoreCase(Config.DUMP_END)) {
        ourSem.release();
      } else {
        String[] parts = msg.split(" => ");
        if (parts.length == 2) {
          String val = parts[1];
          // strip quotes
          val = val.substring(1, val.length() - 1);
          ourMap.put(parts[0], val);
        }
      }
    }
  }
}
