/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.wrapper;

import amip.api.wrapper.callbacks.EventCallback;
import amip.api.wrapper.callbacks.MessageCallback;

/**
 * This is a plain JNI wrapper for the ac.dll. You only need to use it if you want to migrate application using ac.dll
 * to Java. In all the other cases please consider using classes from amip.api.highlevel package. For the documentation
 * of methods in this class please see ac.h file available in AMIP SDK.
 * <p/>
 * Note: don't use methods from this class together with highlevel API.
 */
public class AMIPAPI {
  public static final boolean isAMD64 = "amd64".equals(System.getProperty("os.arch"));

  public static MessageCallback msgCallback;
  public static EventCallback evtCallback;

  /* Use this method to check if native libraries can be loaded */
  public static boolean loadNativeLibs() {
    try {
      if (isAMD64) {
        System.loadLibrary("ac64");
        System.loadLibrary("ac_java64");
      } else {
        System.loadLibrary("ac");
        System.loadLibrary("ac_java");
      }
    } catch (UnsatisfiedLinkError e) {
      return false;
    } catch (SecurityException e1) {
      return false;
    }
    return true;
  }

  static {
    loadNativeLibs();
  }

  public static void setMsgCallback(MessageCallback msgCallback) {
    AMIPAPI.msgCallback = msgCallback;
    register_msg_callback();
  }

  public static void setEvtCallback(EventCallback evtCallback) {
    AMIPAPI.evtCallback = evtCallback;
    register_evt_callback();
  }

  public static native int init_client(String host, int port, int timeout, int dsec, int dcount);

  public static native int init_server(String host, int port);

  public static native void init(int mode);

  public static native void uninit();

  public static native void uninit_client();

  public static native void stop_server();

  public static native int exec(String cmd);

  public static native String eval(String cmd);

  public static native String format(String cmd);

  private static native void register_msg_callback();

  private static native void register_evt_callback();

  /** Invoked from native code */
  public static void message_callback(String msg) {
    if (msgCallback != null) msgCallback.process(msg);
  }

  /** Invoked from native code */
  public static void event_callback(int code) {
    if (evtCallback != null) evtCallback.process(code);
  }

  public static native int add_event_listener(String host, int port, int timeout, int flags, int fail_count);

  public static native int remove_event_listener(String host, int port);

  public static native boolean pingServer(String host, int port, int timeout);

  public static native int get_pl();

  public static native int get_plsize();

  public static native String get_title(int idx);

  public static native void setDestPort(int port);

  public static native void setDestHost(String host);

  public static native void setTimeout(int ms);

  public static native void setSrcPort(int port);

  public static native void setSrcHost(String host);

  public static native void rehash();

  /**
   * Gets last error from ac.dll
   *
   * @return error code
   */
  public static native int get_last_error();

  // Methods to get native API version
  /** @deprecated use version_major */
  public static native int ac_version_major();

  /** @deprecated use version_minor */
  public static native int ac_version_minor();

  /** @deprecated use is_server_started */
  public static native int ac_is_server_started();

  /** @deprecated use is_client_initialized */
  public static native int ac_is_client_initialized();

  public static int version_major() {
    return ac_version_major();
  }

  public static int version_mainor() {
    return ac_version_minor();
  }

  public static int is_server_started() {
    return ac_is_server_started();
  }

  public static int is_client_initialized() {
    return ac_is_client_initialized();
  }

  public static native String get_file_version(String file);
}
