/*
 * Copyright 2008, Sergey Baranov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package amip.api.examples.util;

import sun.misc.Signal;
import sun.misc.SignalHandler;

/**
 * Helper class for examples
 */
public class Helpers {
  /**
   * This method locks current thread, the execution will continue only when InterruptedException is raised, before using
   * this method you should set up signal handler which will interrupt waiting.
   */
  public static void waitUntilTerminated() {
    Object sync = new Object();
    synchronized (sync) {
      try {
        //noinspection WaitNotInLoop
        sync.wait();
      } catch (InterruptedException exc) {
        System.out.println("Ctrl+C pressed. Terminating...");
      }
    }
  }

  /**
   * Sets up a handler for INT signal, from console you can press Ctrl+C to activate the handler. Thread is interrupted
   * when signal is received.
   *
   * @param thread a thread to interrupt.
   */
  public static void setupTerminationHandler(final Thread thread) {
    Signal.handle(
      new Signal("INT"),
      new SignalHandler() {
        public void handle(Signal sig) {
          thread.interrupt();
        }
      }
    );
  }

  /**
   * This method combines the calls to {@link #setupTerminationHandler} and {@link #waitUntilTerminated()}.
   *
   * @param thread a thread to lock and then release upon INT signal (Ctrl+C).
   */
  public static void setupTerminationHandlerAndWait(Thread thread) {
    setupTerminationHandler(thread);
    waitUntilTerminated();
  }
}
