unit AMIPWrapper;

{------------------------------------------------------------------------------}
{                *** Delphi wrapper for the AMIP public API ***                }
{                                                                              }
{ version: 1.2.0                                                                 }
{ Based on the original ac.h file by Serge Baranov.                            }
{                                                                              }
{ by: blert ( blert at blertdiscorporated dot com )                            }
{ last update: January 28, 2006                                                    }
{                                                                              }
{ usage:                                                                       }
{  - Set the DLLPath property to the location of the ac.dll file.              }
{     example: 'x:\path\to\ac.dll'                                             }
{     This is optional and if left blank, or the dll cannot be found at the    }
{     location, the application folder will be searched. If the dll is not     }
{     found in the application folder the registry will be searched for the    }
{     path to the AMIPConfigurator and that folder will be searched for the    }
{     dll.                                                                     }
{  - Call the LoadDLL function.                                                }
{     This will attempt to load ac.dll, if found, and return a TDLLLoadError.  }
{  - Do your thing.                                                            }
{     See the AMIP SDK for the usage of the ac_* functions and procedures.     }
{  - Call the UnloadDLL function.                                              }
{                                                                              }
{------------------------------------------------------------------------------}

interface

uses
  Windows, Classes, SysUtils;

const
  AC_BUFFER_SIZE = 2048;
  // flags for event listener
  AC_EVT_PLAY = $0001;
  AC_EVT_PAUSE = $0002;
  AC_EVT_STOP = $0004;
  AC_EVT_START = $0008;
  AC_EVT_EXIT = $0010;
  AC_EVT_TIMER = $0020;
  AC_EVT_MSG = $0040;
  AC_EVT_CHANGE = $0080;
  AC_EVT_PLCHANGE = $0100;
  AC_EVT_PLREADY = $0200;
  AC_EVT_ALL = AC_EVT_PLAY + AC_EVT_PAUSE + AC_EVT_STOP + AC_EVT_START + AC_EVT_EXIT + AC_EVT_CHANGE + AC_EVT_PLCHANGE + AC_EVT_PLREADY;

type
  TDLLLoadError = (
    ERR_DLL_NOERROR,
    ERR_DLL_ALREADYLOADED,
    ERR_DLL_NOTFOUND,
    ERR_DLL_LOADFAILURE );

  Tac_StartMode = (
    AC_START_ALL,
    AC_START_CLIENT,
    AC_START_SERVER,
    AC_START_NONE  );

  Tac_ErrorCode = (
    AC_ERR_NOERROR,
    AC_ERR_CLIENTISNULL,
    AC_ERR_EXCEPTION,
    AC_ERR_CONNECTIONFAILED,
    AC_ERR_SERVERNOTRUNNING  );

  Tac_msg_callback = procedure(msg: pchar) stdcall;
  Tac_evt_callback = procedure(evt: integer) stdcall;

  TAMIPWrapper = class(TComponent)
  private
    DLLLoaded: boolean;
    DLLHandle: THandle;
    fDLLPath: TFileName;
    procedure SetDLLPath(const aPath: TFileName);
    function FindDLL: boolean;
  protected
    { Protected declarations }
  public
    ac_init_client: function(host: pchar; port, timeout, dsec, dcount: integer): integer; stdcall;
    ac_init_server: function(host: pchar; port: integer): integer; stdcall;
    ac_init: procedure(mode: Tac_StartMode = AC_START_ALL); stdcall;
    ac_uninit: procedure; stdcall;
{}    ac_uninit_client: procedure; stdcall;
{}    ac_stop_server: procedure; stdcall;
    ac_exec: function(const cmd: pchar): Tac_ErrorCode; stdcall;
    ac_eval: function(const cmd: pchar; res: pchar): Tac_ErrorCode; stdcall;
{}    ac_format: function(const cmd: pchar; res: pchar): Tac_ErrorCode; stdcall;
    ac_register_msg_callback: procedure(_1: Tac_msg_callback); stdcall;
    ac_register_evt_callback: procedure(_1: Tac_evt_callback); stdcall;
    ac_add_event_listener: function(const host: pchar; port, timeout: integer; flags: word; fail_count: word = 1): Tac_ErrorCode; stdcall;
    ac_remove_event_listener: function(const host: pchar; port: integer): Tac_ErrorCode; stdcall;
    ac_pingServer: function(const host: pchar; port, timeout: integer): boolean; stdcall;
    { playlist functions }
    ac_get_pl: function: Tac_ErrorCode; stdcall;
    ac_get_plsize: function: integer; stdcall;
    ac_get_title: function(idx: word; buff: pchar): integer; stdcall;
    { client config }
    ac_setDestPort: procedure(port: integer); stdcall;
    ac_setDestHost: procedure(const host: pchar); stdcall;
    ac_setTimeout: procedure(ms: integer); stdcall;
    ac_setSrcPort: procedure(port: integer); stdcall;
    ac_setSrcHost: procedure(const host: pchar); stdcall;
    ac_rehash: procedure; stdcall;
{}    ac_version_major: function: integer; stdcall;
{}    ac_version_minor: function: integer; stdcall;
    function LoadDLL: TDLLLoadError;
    function UnloadDLL: boolean;    
  published
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
    property DLLPath: TFileName read fDLLPath write SetDLLPath;
  end;

procedure Register;

implementation

uses
  Registry;

procedure Register;
begin
  RegisterComponents('blert Tools', [TAMIPWrapper]);
end;

constructor TAMIPWrapper.Create(Owner: TComponent);
begin
  inherited;
  DLLLoaded := false;
end;

destructor TAMIPWrapper.Destroy;
begin

  inherited;
end;

function TAMIPWrapper.FindDLL: boolean;
var
  reg: TRegistry;
  pathBuff: string;
  pathParts: TStringList;
begin
  Result := false;
  //determine if DLLPath holds a valid path to ac.dll
  if fDLLPath <> '' then
    begin
      pathParts := TStringList.Create;
        try
          pathParts.Delimiter := '\';
          pathParts.DelimitedText := fDLLPath;
          if pathParts.Strings[pathParts.Count - 1] = 'ac.dll' then
            if FileExists(fDLLPath) then
              begin
                Result := true;
                Exit;
              end;
        finally
          FreeAndNil(pathParts);
        end;
    end;
  //determine if ac.dll resides in the application folder
  if FileExists('ac.dll') then
    begin
      Result := true;
      SetDLLPath('ac.dll');
      Exit;
    end;
  //try to locate ac.dll in the AMIPConfigurator folder
  reg := TRegistry.Create;
  try
    reg.RootKey := HKEY_LOCAL_MACHINE;
    if reg.OpenKey('\SOFTWARE\AMIPConfigurator', false) then
      begin
        pathBuff := Reg.ReadString('') + '\ac.dll';
        reg.CloseKey;
        if FileExists(pathBuff) then
          begin
            Result := true;
            SetDLLPath(pathBuff);
          end;
      end;
  finally
    FreeAndNil(reg);
  end;
end;

function TAMIPWrapper.LoadDLL: TDLLLoadError;
begin
  Result := ERR_DLL_NOERROR;
  if DLLLoaded then
    begin
      Result := ERR_DLL_ALREADYLOADED;
      Exit;
    end;
  if FindDLL then
    begin
      DLLHandle := LoadLibrary(PChar(fDLLPath));
      if DLLHandle <> 0 then
        begin
          DLLLoaded := True;
          @ac_init_client := GetProcAddress(DLLHandle,'ac_init_client');
          @ac_init_server := GetProcAddress(DLLHandle,'ac_init_server');
          @ac_init := GetProcAddress(DLLHandle,'ac_init');
          @ac_uninit := GetProcAddress(DLLHandle,'ac_uninit');
{}          @ac_uninit_client := GetProcAddress(DLLHandle,'ac_uninit_client');
{}          @ac_stop_server := GetProcAddress(DLLHandle,'ac_stop_server');
          @ac_exec := GetProcAddress(DLLHandle,'ac_exec');
          @ac_eval := GetProcAddress(DLLHandle,'ac_eval');
{}          @ac_format := GetProcAddress(DLLHandle,'ac_format');
          @ac_register_msg_callback := GetProcAddress(DLLHandle,'ac_register_msg_callback');
          @ac_register_evt_callback := GetProcAddress(DLLHandle,'ac_register_evt_callback');
          @ac_add_event_listener := GetProcAddress(DLLHandle,'ac_add_event_listener');
          @ac_remove_event_listener := GetProcAddress(DLLHandle,'ac_remove_event_listener');
          @ac_pingServer := GetProcAddress(DLLHandle,'ac_pingServer');
          @ac_get_pl := GetProcAddress(DLLHandle,'ac_get_pl');
          @ac_get_plsize := GetProcAddress(DLLHandle,'ac_get_plsize');
          @ac_get_title := GetProcAddress(DLLHandle,'ac_get_title');
          @ac_setDestPort := GetProcAddress(DLLHandle,'ac_setDestPort');
          @ac_setDestHost := GetProcAddress(DLLHandle,'ac_setDestHost');
          @ac_setTimeout := GetProcAddress(DLLHandle,'ac_setTimeout');
          @ac_setSrcPort := GetProcAddress(DLLHandle,'ac_setSrcPort');
          @ac_setSrcHost := GetProcAddress(DLLHandle,'ac_setSrcHost');
          @ac_rehash := GetProcAddress(DLLHandle,'ac_rehash');
{}          @ac_version_major := GetProcAddress(DLLHandle,'ac_version_major');
{}          @ac_version_minor := GetProcAddress(DLLHandle,'ac_version_minor');
        end
      else
        begin
          DLLLoaded := False;
          Result := ERR_DLL_LOADFAILURE;
        end;
    end
  else
    begin
      DLLLoaded := False;
      Result := ERR_DLL_NOTFOUND;
    end;
end;

function TAMIPWrapper.UnloadDLL: boolean;
begin
  Result := FreeLibrary(DLLHandle);
end;

procedure TAMIPWrapper.SetDLLPath(const aPath: TFileName);
begin
  fDLLPath := aPath;
end;

{------------------------------------------------------------------------------}
{                              *** Change Log ***                              }
{                                                                              }
{ v 1.1                                                                        }
{   - added playlist functions                                                 }
{                                                                              }
{ v 1.0                                                                        }
{   - first incarnation                                                        }
{                                                                              }
{------------------------------------------------------------------------------}

end.
