using System;
using System.Collections.Generic;
using System.Text;
using System.Threading;

namespace DJMatty.AMIP.ClientWrapper
{
  /// <summary>
  /// AMIPClient is the class that provides integration with the AMIP SDK
  /// </summary>
  public class AMIPClient : IDisposable
  {
    #region Private Members
    private bool _isDisposed = false;
    private uint _playlistSize = 0;
    private bool _playlistRetrieved = false;
    private bool _handlePlaylists = false;
    private Dictionary<string, AMIPServer> _servers = new Dictionary<string, AMIPServer>();
    private Mutex _playlistMutex = new Mutex(false);
    #endregion

    #region Public Delegates
    /// <summary>
    /// AMIPMessage delegate is called when AMIPClient receives a message from AMIP
    /// </summary>
    /// <param name="msg">Text of the message</param>
    public delegate void AMIPMessage(string msg);
    /// <summary>
    /// AMIPEvent delegate is called when AMIPClient receives an event from AMIP
    /// </summary>
    /// <param name="evt">Events received</param>
    public delegate void AMIPEvent(Constants.EventFlags evt);
    #endregion

    #region Public Events
    /// <summary>
    /// MessageReceived event, attach your AMIPMessage delegate to this event
    /// </summary>
    public static event AMIPMessage MessageReceived;
    /// <summary>
    /// EventReceived event, attach your AMIPEvent delegate to this event
    /// </summary>
    public static event AMIPEvent EventReceived;
    #endregion

    #region Public Properties
    /// <summary>
    /// Boolean to indicate that AMIPClient has retrieved a playlist from AMIP
    /// </summary>
    public bool PlaylistRetrieved
    {
      get
      {
        return _playlistRetrieved;
      }
    }
    
    /// <summary>
    /// Major version property
    /// </summary>
    public int MajorVersion
    {
      get
      {
        return AMIPAPI.GetMajorVersion();
      }
    }

    /// <summary>
    /// Minor version property
    /// </summary>
    public int MinorVersion
    {
      get
      {
        return AMIPAPI.GetMinorVersion();
      }
    }

    /// <summary>
    /// Version property, string representation of Major and Minor versions
    /// </summary>
    public string Version
    {
      get
      {
        return String.Format("{0}.{1}", MajorVersion, MinorVersion);
      }
    }

    /// <summary>
    /// Gets/Sets the Source port
    /// </summary>
    public int SourcePort
    {
      get
      {
        return AMIPAPI.GetSourcePort();
      }
      set
      {
        AMIPAPI.SetSourcePort(value);
      }
    }

    /// <summary>
    /// Gets/Sets the Destination port
    /// </summary>
    public int DestinationPort
    {
      get
      {
        return AMIPAPI.GetDestinationPort();
      }
      set
      {
        AMIPAPI.SetDestinationPort(value);
      }
    }

    /// <summary>
    /// Gets/Sets the Source host
    /// </summary>
    public string SourceHost
    {
      get
      {
        StringBuilder host = new StringBuilder(Constants.AC_BUFFER_SIZE);
        AMIPAPI.GetSourceHost(host);
        return host.ToString();
      }
      set
      {
        AMIPAPI.SetSourceHost(value);
      }
    }

    /// <summary>
    /// Gets/Sets the Destination host
    /// </summary>
    public string DestinationHost
    {
      get
      {
        StringBuilder host = new StringBuilder(Constants.AC_BUFFER_SIZE);
        AMIPAPI.GetDestinationHost(host);
        return host.ToString();
      }
      set
      {
        AMIPAPI.SetDestinationHost(value);
      }
    }

    /// <summary>
    /// Sets the timeout
    /// </summary>
    public int Timeout
    {
      set
      {
        AMIPAPI.SetTimeout(value);
      }
    }

    /// <summary>
    /// Gets the playlist size
    /// </summary>
    public uint PlaylistSize
    {
      get
      {
        _playlistSize = (uint)AMIPAPI.GetPlaylistSize();
        return _playlistSize;
      }
    }

    #endregion
    #region Constructor - Destructor
    /// <summary>
    /// Constructor
    /// </summary>
    /// <param name="host">IP Address that AMIP is listening on</param>
    /// <param name="port">Port that AMIP is listening on</param>
    /// <param name="timeout">Timeout in ms for activities</param>
    /// <param name="dsec">Number of seconds to suspend connections for if failed to connect dcount times</param>
    /// <param name="dcount">Number of connection failures allowed before suspension</param>
    /// <param name="handlePlaylists">Set to true to allow AMIPClient to retrieve playlists when they change</param>
    public AMIPClient(string host, int port, int timeout, int dsec, int dcount, bool handlePlaylists)
    {
      _handlePlaylists = handlePlaylists;

      int err = AMIPAPI.InitClient(host, port, timeout, dsec, dcount);
      if (err != 1)
      {
        throw new AMIPAPIException("Error initializing client!");
      }
    }

    /// <summary>
    /// Destructor - DO NOT CALL, called by the gc if the object is not already disposed
    /// </summary>
    ~AMIPClient()
    {
      Dispose(false);
    }

    #endregion

    #region Public Methods
    /// <summary>
    /// Initialise a server used for receiving events/messages from AMIP
    /// </summary>
    /// <param name="host">IP Address to listen on</param>
    /// <param name="port">Port to listen on</param>
    public void InitServer(string host, int port)
    {
      AMIPServer server = new AMIPServer(host, port);

      if (_servers.ContainsKey(server.Key))
      {
        throw new AMIPException("Already a server on this host and port!");
      }

      int err = AMIPAPI.InitServer(server.Host, server.Port);
      if (err != 1)
      {
        throw new AMIPAPIException("Error initializing server!");
      }

      _servers.Add(server.Key, server);

      if (_servers.Count == 1)
      {
        AMIPAPI.RegisterEvtCallback(new Constants.AC_EVT_CALLBACK(EventCallBack));
        AMIPAPI.RegisterMsgCallback(new Constants.AC_MSG_CALLBACK(MessageCallBack));

        if (_handlePlaylists)
        {
          if (Convert.ToInt32(Eval("var_ll")) != Convert.ToInt32(Eval("var_indexed")))
          {
            Exec("reindexq");
          }
          else
          {
            EventCallBack((int)Constants.EventFlags.PlaylistReady);
          }
        }
      }
    }

    /// <summary>
    /// Set the server to listen for events
    /// </summary>
    /// <param name="host">IP Address of the server</param>
    /// <param name="port">Port of the server</param>
    /// <param name="timeout">Timeout</param>
    /// <param name="events">Events to listen for, bitwise or the EventFlags enum members together</param>
    /// <param name="failureCount">Number of times to fail before AMIP removes the listener</param>
    public void ListenForEvents(string host, int port, int timeout, Constants.EventFlags events, uint failureCount)
    {
      string key = AMIPServer.BuildKey(host, port);

      if (_handlePlaylists)
      {
        events |= (Constants.EventFlags.PlaylistChange | Constants.EventFlags.PlaylistReady);
      }

      if (_servers.ContainsKey(key))
      {
        _servers[key].AddEventListener(timeout, events, failureCount);
      }
      else
      {
        throw new AMIPException("Initialize server before adding an event listener");
      }
    }

    /// <summary>
    /// Set the server to listen for events, failure count defaults to 1
    /// </summary>
    /// <param name="host">IP Address of the server</param>
    /// <param name="port">Port of the server</param>
    /// <param name="timeout">Timeout</param>
    /// <param name="events">Events to listen for, bitwise or the EventFlags enum members together</param>
    public void ListenForEvents(string host, int port, int timeout, Constants.EventFlags events)
    {
      ListenForEvents(host, port, timeout, events, 1);
    }

    /// <summary>
    /// Stops the server
    /// </summary>
    public void StopServer()
    {
      AMIPAPI.StopServer();
    }

    /// <summary>
    /// Evaluates AMIP's variable and returns the result.
    /// command can be var_[variable], where variable is any of the AMIP variables
    /// without %, for instance %name becomes var_name. Also cfg_[parameter]
    /// variables are supported (cfg_enabled, cfg_rmienabled, etc.)
    /// Basically, all the $dde variables from help can be evaluated via this
    /// function ('$dde mplug format "%name"' becomes 'format "%name"')
    /// </summary>
    /// <param name="command">Variable to be evaluated</param>
    /// <returns>Results of the evaluation</returns>
    public string Eval(string command)
    {
      StringBuilder result = new StringBuilder(Constants.AC_BUFFER_SIZE);
      Constants.ErrorCode err = AMIPAPI.Eval(command, result);

      if (err != Constants.ErrorCode.AC_ERR_NOERROR)
      {
        throw new AMIPAPIException(err);
      }

      return result.ToString();
    }

    /// <summary>
    /// Passes command to AMIP. For the list of commands see AMIP Help.
    /// Remove '/dde mplug' prefix to get the real command, for instance
    /// in the help you see '/dde mplug announce preset 1' command, this
    /// function will accept 'announce preset 1' as the parameter
    /// </summary>
    /// <param name="command">Command that AMIP is to execute</param>
    public void Exec(string command)
    {
      Constants.ErrorCode err = AMIPAPI.Exec(command);
      if (err != Constants.ErrorCode.AC_ERR_NOERROR)
      {
        throw new AMIPAPIException(err);
      }
    }

    /// <summary>
    /// Same as Eval but takes a format spec string and evaluates it all, the format
    /// spec may look like "%1 - %2 (%br~kbps)"
    /// </summary>
    /// <param name="formatSpec">Format spec to evaluate</param>
    /// <returns>Results of the evaluated format spec</returns>
    public string Format(string formatSpec)
    {
      StringBuilder result = new StringBuilder(Constants.AC_BUFFER_SIZE);
      Constants.ErrorCode err = AMIPAPI.Format(formatSpec, result);
      if (err != Constants.ErrorCode.AC_ERR_NOERROR)
      {
        throw new AMIPAPIException(err);
      }

      return result.ToString();
    }

    /// <summary>
    /// Pings an AMIP Server on the specified address
    /// </summary>
    /// <param name="host">IP Address</param>
    /// <param name="port">Port</param>
    /// <param name="timeout">Timeout (ms)</param>
    /// <returns>True if an AMIP server is listening on that address</returns>
    public bool PingServer(string host, int port, int timeout)
    {
      return AMIPAPI.PingServer(host, port, timeout);
    }

    /// <summary>
    /// Retrieves the current playlist from AMIP and caches it. Call this after a PlaylistReady or PlaylistChange event and before any other playlist call
    /// Use PlaylistSize to retrieve the size of the playlist
    /// </summary>
    public void GetPlaylist()
    {
      _playlistRetrieved = false;
      Constants.ErrorCode err = AMIPAPI.GetPlaylist();

      if (err != Constants.ErrorCode.AC_ERR_NOERROR)
      {
        throw new AMIPAPIException(err);
      }

      _playlistRetrieved = true;
    }
    
    /// <summary>
    /// Retrieves the title of a track in the playlist. Only call this once the playlist is ready and has been cached
    /// </summary>
    /// <param name="index">Zero-based index of the track to retrieve</param>
    /// <returns>Title</returns>
    public string GetTitle(uint index)
    {
      if (!_playlistRetrieved)
      {
        throw new AMIPException("Playlist has not been retrieved!");
      }

      if (index < _playlistSize)
      {
        StringBuilder title = new StringBuilder(Constants.AC_BUFFER_SIZE);
        int err = AMIPAPI.GetTitle(index, title);

        if (err != 1)
        {
          throw new AMIPException("Title is NULL or index out of bounds");
        }

        return title.ToString();
      }
      else
      {
        throw new AMIPException("Index is out of bounds of current playlist size, read PlaylistSize again first!");
      }
    }

    /// <summary>
    /// Reload the configuration and restart the services
    /// </summary>
    public void Rehash()
    {
      AMIPAPI.Rehash();
    }

    /// <summary>
    /// Uninitialise the SDK
    /// </summary>
    public void UnInit()
    {
      Dispose();
    }

    /// <summary>
    /// Dispose frees up the AMIP SDK resources
    /// </summary>
    public void Dispose()
    {
      Dispose(true);
      GC.SuppressFinalize(this);
    }

    #endregion

    #region Private Methods
    private void EventCallBack(int evt)
    {
      Constants.EventFlags amipevent = (Constants.EventFlags)evt;

      if (_handlePlaylists)
      {
        if ((amipevent & Constants.EventFlags.PlaylistChange) == Constants.EventFlags.PlaylistChange)
        {
          Exec("reindexq");
          _playlistRetrieved = false;
        }
        else if((amipevent & Constants.EventFlags.PlaylistReady) == Constants.EventFlags.PlaylistReady)
        {
          _playlistMutex.WaitOne();

          try
          {
            GetPlaylist();
          }
          finally
          {
            _playlistMutex.ReleaseMutex();
          }

          if (EventReceived != null)
          {
            EventReceived(amipevent);
          }
        }
        else
        {
          if (EventReceived != null)
          {
            EventReceived(amipevent);
          }
        }
      }
      else
      {
        if (EventReceived != null)
        {
          EventReceived(amipevent);
        }
      }
    }

    private void MessageCallBack(string msg)
    {
      if (MessageReceived != null)
      {
        MessageReceived(msg);
      }
    }

    #endregion

    #region Protected Methods
    /// <summary>
    /// Internal dispose method that frees the AMIP SDK resources, called by Dispose()
    /// </summary>
    /// <param name="disposing">Set to true by Dispose()</param>
    protected virtual void Dispose(bool disposing)
    {
      if (!_isDisposed)
      {
        if (disposing)
        {
          foreach (AMIPServer server in _servers.Values)
          {
            if (server != null)
            {
              server.Dispose();
            }
          }

          AMIPAPI.UnInit();
        }
      }
      _isDisposed = true;
    }
    #endregion
  }
}
