/* Illustrates how to work with playlist */

#include <windows.h>
#include <stdio.h>
#include <process.h>
#include <string>

#include "../ac.h"

#define SERVER_HOST "127.0.0.1"
#define SERVER_PORT 60335
#define AMIP_HOST "127.0.0.1"
#define AMIP_PORT 60333
#define PL_WAIT 5000

HANDLE sem_ready;
// Synchronization is necessary since playlist can be accessed from different threads
CRITICAL_SECTION cs_list;

// Callback function for events. Warning: invoked in separate thread
void CALLBACK event_callback(int e) {
  switch (e) {
    case AC_EVT_PLREADY: {
      EnterCriticalSection(&cs_list);
      int code = ac_get_pl();
      LeaveCriticalSection(&cs_list);
      if (code != AC_ERR_NOERROR) {
        printf("Failed to get playlist: %d\n", code);
      } else {
        printf("Playlist has %d entries:\n\n", ac_get_plsize());
      }
      ReleaseSemaphore(sem_ready, 1, NULL);
      break;
    }

    case AC_EVT_PLCHANGE:
      // reindex on change
      ac_exec("reindexq");
      break;
  }
}

// helper function for evaluating variables returning integers
int eval_int(const char *var) {
  char buff[AC_BUFFER_SIZE];
  int res = -1;
  if (AC_ERR_NOERROR == ac_eval(var, buff)) {
    res = atoi(buff);
  }
  return res;
}

int main(int argc, char **argv) {
  InitializeCriticalSection(&cs_list);
  // Start server
  if (!ac_init_server("0.0.0.0", SERVER_PORT)) {
    printf("Can't start server on specified port, probably it's already busy\n");
    exit(1);
  }
  // Init client
  if (!ac_init_client(AMIP_HOST, AMIP_PORT, 5000, 5, 1)) {
    printf("Client cannot be initialized\n");
    exit(1);
  }

  // Add event listener to be notified on playlist change and ready events
  if (AC_ERR_NOERROR
      != ac_add_event_listener(SERVER_HOST, SERVER_PORT, 5000, AC_EVT_PLCHANGE | AC_EVT_PLREADY)) {
    printf("Failed to add listener, check that AMIP is running and Remote/API Server is enabled on specified port\n");
    ac_uninit();
    exit(1);
  }

  // Register our callback function, it will receive events from AMIP
  ac_register_evt_callback(event_callback);

  // Ask AMIP to reindex playlist quietly if it's not in sync
  if (eval_int("var_ll") != eval_int("var_indexed")) {
    printf("Asking AMIP to rebuild playlist cache\n");
    ac_exec("reindexq");
    // Set lock and wait until playlist is ready (lock is released in the callback)
    sem_ready = CreateSemaphore(NULL, 0, 1, NULL);
    WaitForSingleObject(sem_ready, PL_WAIT);
  } else {
    // if AMIP's cache is in sync with player's playlist, then just simulate AC_EVT_PLREADY event
    printf("Playlist cache is already in sync!\n");
    event_callback(AC_EVT_PLREADY);
  }

  // Here we should have a playlist
  char title[AC_BUFFER_SIZE];

  EnterCriticalSection(&cs_list);
  int size = ac_get_plsize();
  for (int i = 0; i < size; i++) {
    memset(title, 0, AC_BUFFER_SIZE);
    if (ac_get_title(i, title)) {
      printf("%s\n", title);
    }
  }
  LeaveCriticalSection(&cs_list);

  ac_remove_event_listener(SERVER_HOST, SERVER_PORT);
  ac_uninit();
  DeleteCriticalSection(&cs_list);

  return 0;
}
