#include "ampy.h"

#define AC_DYNAMIC_LOAD
#include "ac.h"

void ampyPlugin::init() {
  if (loaded) return;
  if (ac_load_dll()) {
    loaded = true;
    ac_init(AC_START_CLIENT);

    addCommand("play", "control play", "start playback");
    addCommand("pause", "control pause", "pause playback");
    addCommand("stop", "control stop", "stop playback");
    addCommand("next", "control >", "next song");
    addCommand("previous", "control <", "previous song");
    addCommand("exit", "control exit", "shut down player");
    addCommand("repeat", "control repeat", "toggle repeat");
    addCommand("shuffle", "control shuffle", "toggle shuffle");
  }
}

void ampyPlugin::addCommand(QString cmd, QString api, QString desc) {
  commands.push_back(AmipCommand(cmd, api, desc));
}

ampyPlugin::~ampyPlugin() {
  if (loaded) {
    ac_uninit();
  }
}

void ampyPlugin::launchItem(QList<InputData>* id, CatItem*) {
  if (!loaded) return;
  
  CatItem result = id->last().getTopResult();
  QString cmd = result.shortName;

  if (id->first().hasLabel(HASH_ampy)) {
    for (int i = 0; i < commands.size(); ++i) {
      AmipCommand acmd = commands.at(i);
      if (cmd.compare(acmd.text_command, Qt::CaseInsensitive) == 0) {
        ac_exec(acmd.api_command.toAscii());
        break;
      }
    }
  } else if (id->first().hasLabel(HASH_amip)) {
    ac_exec(id->last().getText().toAscii());
  } else if (id->first().hasLabel(HASH_play)) {
    QString idx = result.fullPath;
    ac_exec(QString("play " + idx).toAscii());
  }
}


void ampyPlugin::getCatalog(QList<CatItem>* items) {
  if (loaded) {
    items->push_back(CatItem("Player Control", "amp", HASH_ampy, getIcon()));
    items->push_back(CatItem("Play Song", "play", HASH_ampy, getIcon()));
    items->push_back(CatItem("AMIP API Command", "amip", HASH_ampy, getIcon()));
  } else {
    items->push_back(CatItem("Player Control Disabled (can't load ac.dll)", "amp", HASH_ampy, getIcon()));
  }
}


void ampyPlugin::getID(uint* id) {
  *id = HASH_ampy;
}


void ampyPlugin::getName(QString* str) {
  *str = "ampy";
}


void ampyPlugin::getLabels(QList<InputData>* id) {
  if(id->count() != 1) return;
  
  const QString & text = id->first().getText();

  if (text.startsWith("amp", Qt::CaseInsensitive)) {
    id->last().setLabel(HASH_ampy);
  } else if (text.startsWith("play", Qt::CaseInsensitive)) {
    id->last().setLabel(HASH_play);
  } else if (text.startsWith("amip", Qt::CaseInsensitive)) {
    id->last().setLabel(HASH_amip);
  }
}

void ampyPlugin::getResults(QList<InputData>* id, QList<CatItem>* results) {
  if (!loaded) return;
  
  if (id->count() != 2) return;
  
  if (id->first().hasLabel(HASH_ampy)) {
    for (int i = 0; i < commands.size(); ++i) {
      AmipCommand acmd = commands.at(i);
      results->push_front(CatItem(acmd.description, acmd.text_command, HASH_ampy, getIcon()));
    }
  } else if (id->first().hasLabel(HASH_amip)) {
    results->push_front(CatItem("AMIP API command", id->last().getText(), HASH_amip, getIcon()));
  } else if (id->first().hasLabel(HASH_play)) {
    if(ac_get_pl() != AC_ERR_NOERROR) return;
    int size = ac_get_plsize();

    if(size < 1) {
      ac_exec("reindexq");
      if(ac_get_pl() != AC_ERR_NOERROR) return;
      size = ac_get_plsize();
    }
    
    QTextCodec *codec = QTextCodec::codecForLocale();
    QString num;
    for (int i = 0; i < size; i++) {
      char buff[AC_BUFFER_SIZE];
      if(ac_get_title(i, buff)) {
        num.setNum(i+1);
        results->push_front(CatItem(num, codec->toUnicode(buff), HASH_play, getIcon()));
      }
    }
  }
}


QString ampyPlugin::getIcon() {
  return libPath + "/icons/ampy.png";
}

void ampyPlugin::setPath(QString * path) {
  libPath = *path;
}

int ampyPlugin::msg(int msgId, void* wParam, void* lParam) {
  bool handled = false;
  switch (msgId) {
  case MSG_INIT:
    init();
    handled = true;
    break;
  case MSG_GET_ID:
    getID((uint*) wParam);
    handled = true;
    break;
  case MSG_GET_NAME:
    getName((QString*) wParam);
    handled = true;
    break;
  case MSG_GET_LABELS:
    getLabels((QList<InputData>*) wParam);
    handled = true;
    break;
  case MSG_GET_RESULTS:
    getResults((QList<InputData>*) wParam, (QList<CatItem>*) lParam);
    handled = true;
    break;
  case MSG_GET_CATALOG:
    getCatalog((QList<CatItem>*) wParam);
    handled = true;
    break;
  case MSG_PATH:
    setPath((QString *) wParam);
    break;
  case MSG_LAUNCH_ITEM:
    launchItem((QList<InputData>*) wParam, (CatItem*) lParam);
    handled = true;
    break;
  default:
    break;
  }

  return handled;
}


Q_EXPORT_PLUGIN2(ampy, ampyPlugin)
