using System;
using System.Collections.Generic;
using System.Text;
using DJMatty.AMIP.ClientWrapper;

namespace DJMatty.AMIPApplication
{
	public class Test
	{
		private AMIPClient _client = null;

		public void Run()
		{
			// create a new client instance... the handlePlaylists parameter
			// tells the client to process the playlist event messages internally
			// you must ensure that your listener listens for the PlaylistReady
			// and PlaylistChanged events for this to work correctly.
			_client = new AMIPClient("127.0.0.1", 60333, 5000, 5, 1, true);

			// using will force dispose to be called -- dispose will ensure that the AMIP SDK is uninitialised
			// and any server listeners are removed
			using (_client)
			{
				// first lets get the currently playing track name
				try
				{
					Console.WriteLine(String.Format("Now Playing: {0}", _client.Eval("var_s")));
				}
				catch (AMIPAPIException e)
				{
					// AMIPClient will throw either AMIPException, which is an issue with the client
					// or AMIPAPIException (which inherits from AMIPException) which is an error returned
					// from an AMIP SDK call. Message property contains the textual representation of the 
					// Constants.ErrorCode enum
					Console.WriteLine(String.Format("Caught exception when calling eval: {0}", e.ToString()));
				}

				// set up events for receiving AMIP Events and Messages
				try
				{
					AMIPClient.EventReceived += new AMIPClient.AMIPEvent(AMIPClient_EventReceived);
					AMIPClient.MessageReceived += new AMIPClient.AMIPMessage(AMIPClient_MessageReceived);

					// need to set up a server to listen for events on
					_client.InitServer("127.0.0.1", 60336);

					// need to tell amip sdk to listen for events on the server (use the EventFlags enum to
					// specify which events are required, e.g. EventFlags.PlaylistReady | EventFlags.PlaylistChanged)
					// NOTE: because the AMIPClient was constructed with handlePlaylists = true, the client will
					// automatically add the PlaylistReady and PlaylistChanged events to the events you pass
					_client.ListenForEvents("127.0.0.1", 60336, 5000, Constants.EventFlags.All);
				}
				catch (AMIPException e)
				{
					Console.WriteLine(String.Format("Caught exception: {0}", e.ToString()));
				}

				// wait for enter to be pressed
				Console.ReadLine();
			};

			_client = null;
		}


		// dont forget that these methods are called by the thread servicing the AMIP server socket, NOT the UI thread
		// need to use Control.Invoke to update any UI
		void AMIPClient_MessageReceived(string msg)
		{
			Console.WriteLine(String.Format("Message received: '{0}'", msg));
		}

		// dont forget that these methods are called by the thread servicing the AMIP server socket, NOT the UI thread
		// need to use Control.Invoke to update any UI
		void AMIPClient_EventReceived(Constants.EventFlags evt)
		{
			Console.WriteLine(String.Format("Event '{0}' received!", evt));


			if (((evt & Constants.EventFlags.Change) == Constants.EventFlags.Change) || ((evt & Constants.EventFlags.Play) == Constants.EventFlags.Play))
			{
				try
				{
					Console.WriteLine(String.Format("Now Playing: {0}", _client.Eval("var_s")));
				}
				catch (AMIPAPIException e)
				{
					Console.WriteLine(String.Format("Caught exception when calling eval: {0}", e.ToString()));
				}
			}

			if ((evt & Constants.EventFlags.PlaylistReady) == Constants.EventFlags.PlaylistReady)
			{
				uint pls = _client.PlaylistSize;
				Console.WriteLine(String.Format("Playlist is {0} tracks long", pls));
				for (uint i = 0; i < pls; i++)
				{
					Console.WriteLine(String.Format("Playlist track {0}: '{1}'", i, _client.GetTitle(i)));
				}
			}
		}
	}
}
